using System;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;

namespace HigherLogic.Data
{
	/// <summary>
	/// Provides access to the database using a SqlClient connection and methods
	/// </summary>
	public class DataServer : MarshalByRefObject, IDisposable
	{
		#region Fields
		private string mConnectionString = string.Empty;
		private SqlConnection mConnection;
		private SqlTransaction mTransaction;
		#endregion Fields

		#region Constructors
		/// <summary>
		/// Instantiate a new DataServer using the default connection string found in the config
        /// file, in the parameter DefaultConnection
		/// </summary>
		public DataServer()
		{
            // grab the connection string from Web.Config.
            mConnectionString = ConfigurationManager.ConnectionStrings["AMSConnection"].ConnectionString;
        }

		/// <summary>
		/// Instantiate a new DataServer using the supplied connection string
		/// </summary>
		/// <param name="connectionString">
		/// A standard SQL Client connection string like that which would be passed to <see cref="System.Data.SqlClient.SqlConnection"/>.
		/// </param>
		/// <example>Sample value for connectionString:<para />
		/// "Data Source=DBServer;User ID=sa;Password=pw;Initial Catalog=HHR; Network Library=dbmssocn"</example>
		public DataServer(string connectionString)
		{
			mConnectionString = connectionString;
		}

		#endregion Constructors

		#region Properties
		/// <summary>
		/// Gets the current <see cref="System.Data.SqlClient.SqlConnection"/> to the database.
		/// </summary>
		/// <value>A <see cref="System.Data.SqlClient.SqlConnection"/> object to be used by this instance of the DataServer
		/// to interact with the database.</value>
		/// <remarks>If there is not currently a connection a new one will be created and returned.</remarks>
		public SqlConnection Connection
		{
			get 
			{
				if(mConnection == null)
					mConnection = new SqlConnection(mConnectionString);
				return mConnection;
			}
		}

		/// <summary>
		/// Gets the SqlClient connection string assigned by the constructor.
		/// </summary>
		/// <value>A standard SQL Client connection string like that which would be passed to <see cref="System.Data.SqlClient.SqlConnection"/>.</value>
		/// <example>"Data Source=DBServer;User ID=sa;Password=pw;Initial Catalog=HHR; Network Library=dbmssocn"</example>
		internal string ConnectionString
		{
			get
			{
				return mConnectionString;
			}
		}

		#endregion

		#region Destructor

		/// <summary>
		/// Destructor
		/// </summary>
		~DataServer()
		{
			Dispose(false);
		}

		#endregion Destructor

		#region Members

		/// <summary>
		/// Starts a database transaction.  Any other database operations performed using this instance of the DataServer
		/// will be in the same transaction until a <see cref="CommitTransaction"/> or <see cref="RollbackTransaction"/> is called.
		/// </summary>
		public void BeginTransaction()
		{
			if(mTransaction == null)
			{
				OpenConnection();
				mTransaction = Connection.BeginTransaction();
			}
		}

		/// <summary>
		/// Binds the supplied parameters to the supplied command and executes the command.  The command must have been
		/// prepared with a call to <see cref="PrepareCommand"/>.  This method can then be called in a loop with different
		/// parameters to save the prepare time when the same statement is executed over and over.
		/// </summary>
		/// <param name="command">The SqlCommand to use to execute the SQL.  Hold on to this to use repetatively</param>
		/// <param name="commandParameters">The parameters to be bound to the query</param>
		public static void BindAndExecute(SqlCommand command, SqlParameter[] commandParameters)
		{
			for (int i=0,ic=commandParameters.Length;i<ic;i++)
				command.Parameters[i] = commandParameters[i];
			command.ExecuteNonQuery();
		}

		/// <summary>
		/// Closes the connection to the database if it is open.  If there is no open Connection no harm is done by 
		/// calling this method. If there is an open connection and there is also an open transaction the transaction 
		/// will be rolled back before the connection is closed.
		/// </summary>
		public void CloseConnection()
		{
			if(mConnection != null)
			{
				ConnectionState state = mConnection.State;
				if(state == ConnectionState.Open)
				{
					RollbackTransaction();
					mConnection.Close();
				}
				else if(state == ConnectionState.Broken)
					mConnection.Close();
			}
			mTransaction = null;
		}

		/// <summary>
		/// If there is an open transaction associated with this DataServer instance, it will be committed.  If
		/// there is no open transaction no harm is done by calling this method.
		/// </summary>
		public void CommitTransaction()
		{
			if(mTransaction != null)
			{
				mTransaction.Commit();
				mTransaction = null;
			}
		}

		/// <summary>
		/// Rolls back and open transaction and closes any open connection
		/// </summary>
		public void Dispose()
		{
			Dispose(true);
			GC.SuppressFinalize(this);
		}

		// Rolls back and open transaction and closes any open connection
		private void Dispose(bool disposing)
		{
			if(disposing)
			{
				if(mTransaction != null)
				{
					try
					{
						mTransaction.Rollback();
						mTransaction.Dispose();
					}
					catch{}	
				}
				if(mConnection != null)
				{	
					try
					{
						mConnection.Close();
						mConnection.Dispose();
					}
					catch{}						
				}
			}     
			mTransaction = null;
			mConnection = null;
		}

		/// <summary>
		/// Executes the SQL query in the sql parameter against the database.  Use this method when the sql query
		/// does not return a result set (e.g., INSERT, UPDATE, or some stored procs).  Otherwise, use 
		/// <see cref="ExecuteQuery"/> if a result set is desired.
		/// </summary>
		/// <param name="commandType">The type of cammand contained in the sql parameter.</param>
		/// <param name="sql">The SQL command to execute</param>
		/// <returns>The number of rows affected by the query</returns>
		public int ExecuteNonQuery(CommandType commandType, string sql)
		{
			return ExecuteNonQuery(commandType, sql, null);
		}

		/// <summary>
		/// Executes the SQL query in the sql parameter against the database.  Use this method when the sql query
		/// does not return a result set (e.g., INSERT, UPDATE, or some stored procs).  Otherwise, use 
		/// <see cref="ExecuteQuery"/> if a result set is desired.
		/// </summary>
		/// <param name="commandType">The type of cammand contained in the sql parameter.</param>
		/// <param name="sql">The SQL command to execute</param>
		/// <param name="commandParameters">The parameters to be bound to the query</param>
		/// <returns>The number of rows affected by the query</returns>
		public int ExecuteNonQuery(CommandType commandType, string sql, SqlParameter[] commandParameters)
		{	
			int rowsAffected = 0;
			bool closeConnection = false;

			try
			{
				if (mConnection != null && mConnection.State == ConnectionState.Open && mTransaction == null)
					closeConnection = true;
				SqlCommand command = new SqlCommand();
				PrepareCommand(command, commandType, sql, commandParameters);
				rowsAffected = command.ExecuteNonQuery();
				RetrieveParameters(command, commandParameters);
			}
			finally
			{
				try
				{
					if (closeConnection)
						CloseConnection();
				}
				catch{}
			}
			return rowsAffected;
		}

		/// <summary>
		/// Executes the SQL query in the sql parameter against the database.  Use this method when the sql query
		/// returns a result set (e.g., SELECT or some stored procs).  Otherwise, use <see cref="ExecuteNonQuery"/>.
		/// </summary>
		/// <param name="commandType">The type of cammand contained in the sql parameter.</param>
		/// <param name="sql">The SQL command to execute</param>
		/// <returns>A Dataset containing the results of the SQL</returns>
		public DataSet ExecuteQuery(CommandType commandType, string sql)
		{
			return ExecuteQuery(commandType, sql, null);
		}

		/// <summary>
		/// Executes the SQL query in the sql parameter against the database.  Use this method when the sql query
		/// returns a result set (e.g., SELECT or some stored procs).  Otherwise, use <see cref="ExecuteNonQuery"/>.
		/// </summary>
		/// <param name="commandType">The type of cammand contained in the sql parameter.</param>
		/// <param name="sql">The SQL command to execute</param>
		/// <param name="commandParameters">The parameters to be bound to the query</param>
		/// <returns>A Dataset containing the results of the SQL</returns>
		public DataSet ExecuteQuery(CommandType commandType, string sql, SqlParameter[] commandParameters)
		{	
			DataSet ds = null;
			bool closeConnection = false;

			try
			{
				if (mConnection != null && mConnection.State == ConnectionState.Open && mTransaction == null)
					closeConnection = true;
				ds = new DataSet();
				SqlCommand command = new SqlCommand();
				PrepareCommand(command, commandType, sql, commandParameters);
				SqlDataAdapter da = new SqlDataAdapter();
				da.SelectCommand = command;
				da.Fill(ds);
			}
			finally
			{
				if(closeConnection)
				{
					try
					{
						CloseConnection();
					}
					catch{}
				}
			}
			return ds;
		}

		/// <summary>
		/// Executes the SQL query in the sql parameter against the database.  Use this method when the sql query
		/// returns a result set (e.g., SELECT or some stored procs).  Otherwise, use <see cref="ExecuteNonQuery"/>.
		/// </summary>
		/// <param name="commandType">The type of cammand contained in the sql parameter.</param>
		/// <param name="sql">The SQL command to execute</param>
		/// <returns>A SqlDataReader containing the results of the SQL statement</returns>
		public SqlDataReader ExecuteReader(CommandType commandType, string sql)
		{
			return ExecuteReader(commandType, sql, null);
		}

		/// <summary>
		/// Executes the SQL query in the sql parameter against the database.  Use this method when the sql query
		/// returns a result set (e.g., SELECT or some stored procs).  Otherwise, use <see cref="ExecuteNonQuery"/>.
		/// </summary>
		/// <param name="commandType">The type of cammand contained in the sql parameter.</param>
		/// <param name="sql">The SQL command to execute</param>
		/// <param name="commandParameters">The parameters to be bound to the query</param>
		/// <returns>A SqlDataReader containing the results of the SQL statement</returns>
		public SqlDataReader ExecuteReader(CommandType commandType, string sql, SqlParameter[] commandParameters)
		{	
			SqlDataReader reader = null;
			bool closeConnection = false;

			try
			{
				if (mConnection != null && mConnection.State == ConnectionState.Open && mTransaction == null)
					closeConnection = true;
				SqlCommand command = new SqlCommand();
				PrepareCommand(command, commandType, sql, commandParameters);
				if(closeConnection)
				{
					reader = command.ExecuteReader(CommandBehavior.CloseConnection);
				}
				else
				{
					reader = command.ExecuteReader();
				}
			}
			catch
			{
				if(closeConnection)
				{
					try
					{
						CloseConnection();
					}
					catch{}
				}
				throw;
			}
			return reader;
		}

		// opens the connection to the db
		private void OpenConnection()
		{
			SqlConnection connection = Connection;
			if(connection.State == ConnectionState.Closed)
			{
				connection.Open();
			}
			else if(connection.State == ConnectionState.Broken)
			{
				connection.Close();
				connection.Open();
			}
		}

		/// <summary>
		/// Prepares a SQL command to be executed.  This should not normally be called outside of this class 
		/// but is public in case the same statement needs to be executed over and over with different bind variables.
		/// </summary>
		/// <param name="command">The SqlCommand to use to execute the SQL.  Hold on to this to use repetatively</param>
		/// <param name="commandType">The type of cammand contained in the sql parameter.</param>
		/// <param name="sql">The SQL command to execute</param>
		/// <param name="commandParameters">The parameters to be bound to the query</param>
		public void PrepareCommand(SqlCommand command, CommandType commandType, string sql, SqlParameter[] commandParameters)
		{
			OpenConnection();
			command.Connection = Connection;
			command.CommandText = sql;
			command.CommandTimeout = 90;
			if(mTransaction != null)
				command.Transaction = mTransaction;
			command.CommandType = commandType;

			if(commandParameters != null)
				for(int i=0, ic=commandParameters.Length;i<ic;i++)
					command.Parameters.Add(commandParameters[i]);
		}

		// gets any out parameters from a query call and sets them back in the parameters array
		private static void RetrieveParameters (SqlCommand cmd,  params SqlParameter[] commandParameters)
		{
			if (commandParameters != null)
				for (int i=0, ic=commandParameters.Length;i<ic;i++)
					if (commandParameters[i].Direction != ParameterDirection.Input)
						commandParameters[i].Value = cmd.Parameters[i].Value;
		}

		/// <summary>
		/// If there is an open transaction associated with this DataServer instance, all statements executed
		/// within that transaction will be rolled back (discarded).  If there is no open transaction no harm 
		/// is done by calling this method.
		/// </summary>
		public void RollbackTransaction()
		{
			if(mTransaction != null)
			{
				mTransaction.Rollback();
				mTransaction = null;
			}
		}

		#endregion Members
	}
}
