﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.IO;
using System.Security.Cryptography;
using System.Text;
using System.Web.Services;
using Asi.iBO;
using Asi.iBO.ContactManagement;
using Asi.iBO.Errors;
using Asi.iBO.SystemConfig;
using AccessViolationException = System.AccessViolationException;
using DataServer = HigherLogic.Data.DataServer;

/// <summary>
/// Summary description for HigherLogicProfile
/// </summary>
[WebService(Namespace = "http://higherlogic.com/")]
[WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
public class HigherLogicProfile : System.Web.Services.WebService
{
    public HigherLogicProfile()
    {
    }

    private static string Decrypt(string value)
    {
        if (string.IsNullOrEmpty(value))
            return value;

        byte[] keyBytes = Encoding.ASCII.GetBytes(ConfigurationManager.AppSettings["EncryptionKey"]);
        byte[] key = new byte[32];
        byte[] iv = new byte[16];
        Array.Copy(keyBytes, key, 32);
        Array.Copy(keyBytes, 32, iv, 0, 16);

        string decrypted = null;
        RijndaelManaged rm = null;
        try
        {
            rm = new RijndaelManaged();
            rm.KeySize = 256;
            rm.Mode = CipherMode.CBC;
            rm.Key = key;
            rm.IV = iv;
            using (ICryptoTransform decryptor = rm.CreateDecryptor(rm.Key, rm.IV))
            {
                using (MemoryStream ms = new MemoryStream(Convert.FromBase64String(value)))
                {
                    ms.Seek(0, SeekOrigin.Begin);
                    using (CryptoStream cs = new CryptoStream(ms, decryptor, CryptoStreamMode.Read))
                    {
                        using (StreamReader sr = new StreamReader(cs))
                        {
                            decrypted = sr.ReadToEnd();
                            sr.Close();
                        }
                        cs.Close();
                    }
                    ms.Close();
                }
            }
        }
        finally
        {
            if (rm != null)
                rm.Clear();
        }
        return decrypted;
    }

    private static string Encrypt(string value)
    {
        if (string.IsNullOrEmpty(value))
            return null;

        byte[] keyBytes = Encoding.ASCII.GetBytes(ConfigurationManager.AppSettings["EncryptionKey"]);
        byte[] key = new byte[32];
        byte[] iv = new byte[16];
        Array.Copy(keyBytes, key, 32);
        Array.Copy(keyBytes, 32, iv, 0, 16);

        byte[] buffer = null;
        RijndaelManaged rm = null;
        try
        {
            rm = new RijndaelManaged();
            rm.KeySize = 256;
            rm.Mode = CipherMode.CBC;
            rm.Key = key;
            rm.IV = iv;
            using (ICryptoTransform encryptor = rm.CreateEncryptor(rm.Key, rm.IV))
            {
                using (MemoryStream ms = new MemoryStream())
                {
                    using (CryptoStream cs = new CryptoStream(ms, encryptor, CryptoStreamMode.Write))
                    {
                        using (StreamWriter sw = new StreamWriter(cs))
                        {
                            sw.Write(value);
                            sw.Flush();
                            cs.FlushFinalBlock();
                            buffer = ms.ToArray();
                            sw.Close();
                        }
                        cs.Close();
                    }
                    ms.Close();
                }
            }
        }
        finally
        {
            if (rm != null)
                rm.Clear();
        }

        if (buffer != null)
            return Convert.ToBase64String(buffer);
        return null;
    }

    [WebMethod(Description = "Returns the contact's profile data.", EnableSession = false)]
    public HLImisContact GetContactData(string securityKey, string contactId, string webLogin)
    {
        ValidateSecurityKey(securityKey);
        HLImisContact hlc = new HLImisContact();

        try
        {
            InitializeIbo();
            CContactUser user = CContactUser.LoginByWebLogin(webLogin);
            CContact c = new CContact(user, contactId);
            hlc.Load(c);
        }
        catch (Exception excp)
        {
            hlc.ExceptionMessage = excp.Message + excp.StackTrace;
        }

        return hlc;
    }

    private static HLImisKeyValuePair[] GetLookup(string tableName)
    {
        string sql = @"select TABLE_NAME,UPPER_CODE,DESCRIPTION from Gen_Tables where TABLE_NAME = @TableName order by UPPER_CODE";
        SqlParameter[] p = new SqlParameter[1];
        p[0] = new SqlParameter("@TableName", SqlDbType.VarChar);
        p[0].Value = tableName;

        List<HLImisKeyValuePair> list = new List<HLImisKeyValuePair>();

        using (DataServer server = new DataServer())
        {
            using (SqlDataReader reader = server.ExecuteReader(CommandType.Text, sql, p))
            {
                while (reader.Read())
                    list.Add(new HLImisKeyValuePair(NullToEmtpy(reader[1] as string), NullToEmtpy(reader[2] as string)));
            }
        }

        return list.ToArray();
    }

    private static string NullToEmtpy(string value)
    {
        if (value == null)
            return "";
        return value;
    }


    [WebMethod(Description = "Returns the system reference data.", EnableSession = false)]
    public HLImisReference GetReferenceData(string securityKey)
    {
        ValidateSecurityKey(securityKey);
        HLImisReference hlr = new HLImisReference();
        try
        {
            InitializeIbo();

            CReferenceData r = iboAdmin.ReferenceData;
            
            try
            {
                CCountry[] countries = r.Countries;
                hlr.CountryNames = new string[countries.Length];
                for (int x = 0; x <= countries.Length; x++)
                    hlr.CountryNames[x] = countries[x].CountryName;
            }
            catch
            {
                //  Index out of bounds?
            }

            string[] tableNames = r.ExtTableNames;
            if (tableNames != null && tableNames.Length > 0)
            {
                hlr.ExtTables = new HLImisExtTable[tableNames.Length];
                for (int x = 0; x < tableNames.Length; x++)
                {
                    hlr.ExtTables[x] = new HLImisExtTable();
                    hlr.ExtTables[x].Load(r.GetExtTableDefinition(tableNames[x]), true);
                }
            }

            if (r.Prefixes != null)
            {
                hlr.Prefixes = new string[r.Prefixes.Length];
                Array.Copy(r.Prefixes, hlr.Prefixes, r.Prefixes.Length);
            }

            if (r.StatesProvinces != null)
            {
                hlr.StatesProvinces = new string[r.StatesProvinces.Length];
                Array.Copy(r.StatesProvinces, hlr.StatesProvinces, r.StatesProvinces.Length);
            }

            if (r.Suffixes != null)
            {
                hlr.Suffixes = new string[r.Suffixes.Length];
                Array.Copy(r.Suffixes, hlr.Suffixes, r.Suffixes.Length);
            }

            CContactConfig c = iboAdmin.ContactsConfig;

            hlr.Address2Prompt = c.Address2Prompt;
            hlr.Address3Prompt = c.Address3Prompt;
            hlr.AddressPurposes = c.AddressPurposes;
            hlr.ChapterPrompt = c.ChapterPrompt;
            hlr.CountyPrompt = c.CountyPrompt;
            hlr.InstitutePrompt = c.InstitutePrompt;
            hlr.MainAddressPrompt = c.MainAddressPrompt;
            hlr.StatePrompt = c.StatePrompt;
            hlr.TollFreePrompt = c.TollFreePrompt;
            
        }
        catch(Exception excp)
        {
            hlr.ExceptionMessage = hlr.ExceptionMessage + excp.Message + excp.StackTrace;
        }
        return hlr;
    }

    private static void InitializeIbo()
    {
        if (!iboAdmin.IsSystemInitialized)
            iboAdmin.InitializeSystem();
    }

    [WebMethod(Description = "Updates the contact's profile data.", EnableSession = false)]
    public HLImisContact SetContactData(string securityKey, HLImisContact hlc, string webLogin)
    {
        ValidateSecurityKey(securityKey);
        InitializeIbo();

        CContact c = null;
        HLImisContact result;
        StringBuilder sb = new StringBuilder();
        try
        {
            CContactUser user = CContactUser.LoginByWebLogin(webLogin);
            sb.AppendLine(string.Format("Login {0} for {1}", user == null ? "failure" : "success", webLogin));
            DateTime d;
            
            c = new CContact(user, hlc.ContactId);
            sb.AppendLine(string.Format("Contact {0} {1} loaded", hlc.ContactId, c == null ? "not" : ""));
            if (!string.IsNullOrEmpty(hlc.BirthDate) &&
                DateTime.TryParse(hlc.BirthDate, out d) &&
                c.BirthDate != d)
                c.BirthDate = d;
            if (string.Compare(c.Designation, hlc.Designation) != 0)
                c.Designation = hlc.Designation;
            if (string.Compare(c.EmailAddress, hlc.EmailAddress) != 0)
                c.EmailAddress = hlc.EmailAddress;
            if (c.ExcludeFromDirectory != hlc.ExcludeFromDirectory)
                c.ExcludeFromDirectory = hlc.ExcludeFromDirectory;
            if (c.ExcludeFromMailings != hlc.ExcludeFromMailings)
                c.ExcludeFromMailings = hlc.ExcludeFromMailings;
            if (string.Compare(c.Fax, hlc.Fax) != 0)
                c.Fax = hlc.Fax;
            if (string.Compare(c.FirstName, hlc.FirstName) != 0)
                c.FirstName = hlc.FirstName;
            if (string.Compare(c.Gender.ToString(), hlc.Gender) != 0)
                c.Gender = (ContactGender)Enum.Parse(typeof(ContactGender), hlc.Gender);
            if (string.Compare(c.HomePhone, hlc.HomePhone) != 0)
                c.HomePhone = hlc.HomePhone;
            if (string.Compare(c.Informal, hlc.Informal) != 0)
                c.Informal = hlc.Informal;
            if (string.Compare(c.InstituteName, hlc.InstituteName) != 0)
            {
                c.InstituteName = hlc.InstituteName;
                c.InstituteContactId = string.Empty;
            }
            if (string.Compare(c.LastName, hlc.LastName) != 0)
                c.LastName = hlc.LastName;
            if (string.Compare(c.MiddleName, hlc.MiddleName) != 0)
                c.MiddleName = hlc.MiddleName;
            if (string.Compare(c.Prefix, hlc.Prefix) != 0)
                c.Prefix = hlc.Prefix;
            if (string.Compare(c.Suffix, hlc.Suffix) != 0)
                c.Suffix = hlc.Suffix;
            if (string.Compare(c.Title, hlc.Title) != 0)
                c.Title = hlc.Title;
            if (string.Compare(c.TollFreePhone, hlc.TollFreePhone) != 0)
                c.TollFreePhone = hlc.TollFreePhone;
            if (string.Compare(c.WebsiteAddress, hlc.WebsiteAddress) != 0)
                c.WebsiteAddress = hlc.WebsiteAddress;
            if (string.Compare(c.WorkPhone, hlc.WorkPhone) != 0)
                c.WorkPhone = hlc.WorkPhone;
            sb.AppendLine(string.Format("Contact fields complete"));

            if (hlc.Addresses != null && hlc.Addresses.Length > 0)
            {
                foreach (HLImisAddress hla in hlc.Addresses)
                {
                    CAddress ca;
                    if (hla.AddressNumber == 0)
                    {
                        ca = c.NewAddress();
                        sb.AppendLine(string.Format("Create new address"));
                    }
                    else
                    {
                        ca = c.GetAddressByNum(hla.AddressNumber);
                        sb.AppendLine(string.Format("Use address {0}", hla.AddressNumber));
                    }

                    if (ca != null)
                    {
                        if (string.Compare(ca.Address1, hla.Address1) != 0)
                            ca.Address1 = hla.Address1;
                        if (string.Compare(ca.Address2, hla.Address2) != 0)
                            ca.Address2 = hla.Address2;
                        if (string.Compare(ca.Address3, hla.Address3) != 0)
                            ca.Address3 = hla.Address3;
                        if (string.Compare(ca.AddressPurpose, hla.AddressPurpose) != 0)
                            ca.AddressPurpose = hla.AddressPurpose;
                        if (string.Compare(ca.City, hla.City) != 0)
                            ca.City = hla.City;
                        if (string.Compare(ca.Country, hla.Country) != 0)
                            ca.Country = hla.Country;
                        if (string.Compare(ca.County, hla.County) != 0)
                            ca.County = hla.County;
                        if (string.Compare(ca.EmailAddress, hla.EmailAddress) != 0)
                            ca.EmailAddress = hla.EmailAddress;
                        if (string.Compare(ca.Fax, hla.Fax) != 0)
                            ca.Fax = hla.Fax;
                        if (string.Compare(ca.InstituteName, hla.InstituteName) != 0)
                            ca.InstituteName = hla.InstituteName;
                        // Preferred options should only be set when true.  iBO sets other instances to false
                        if (hla.IsPreferredBill && ca.IsPreferredBill != hla.IsPreferredBill)
                            ca.IsPreferredBill = hla.IsPreferredBill;
                        if (hla.IsPreferredMail && ca.IsPreferredMail != hla.IsPreferredMail)
                            ca.IsPreferredMail = hla.IsPreferredMail;
                        if (hla.IsPreferredShip && ca.IsPreferredShip != hla.IsPreferredShip)
                            ca.IsPreferredShip = hla.IsPreferredShip;
                        if (string.Compare(ca.Phone, hla.Phone) != 0)
                            ca.Phone = hla.Phone;
                        if (string.Compare(ca.PostalCode, hla.PostalCode) != 0)
                            ca.PostalCode = hla.PostalCode;
                        if (string.Compare(ca.StateProvince, hla.StateProvince) != 0)
                            ca.StateProvince = hla.StateProvince;
                        if (string.Compare(ca.TollFree, hla.TollFree) != 0)
                            ca.TollFree = hla.TollFree;
                    }
                    else
                    {
                        sb.AppendLine(string.Format("No address"));
                    }
                }
            }
            else
            {
                sb.AppendLine(string.Format("No addresses"));
            }

            if (hlc.ExtTables != null && hlc.ExtTables.Length > 0)
            {
                foreach (HLImisExtTable hlt in hlc.ExtTables)
                {
                    bool create = false;
                    CExtTable ext = null;
                    if (hlt.SequenceNumber == -1)
                    {
                        ext = c.NewExtTableRow(hlt.TableName);
                    }
                    else
                    {
                        for (int x = 0; x < c.ExtTables.Length; x++)
                        {
                            if (string.Compare(c.ExtTables[x].TableName, hlt.TableName) == 0)
                            {
                                if(c.ExtTables[x].InstancesCount == 0)
                                {
                                    ext = c.ExtTables[x];
                                    create = true;
                                    sb.AppendLine(string.Format("Using non-instance of {0}", ext.TableName));
                                }
                                else if (c.ExtTables[x].InstancesCount == 1)
                                {
                                    ext = c.ExtTables[x].GetInstance(0);
                                    sb.AppendLine(string.Format("Using instance 1 of {0}", ext.TableName));
                                }
                                else
                                {
                                    for(int y = 0; y < c.ExtTables[x].InstancesCount; y++)
                                    {
                                        CExtTable cet = c.ExtTables[x].GetInstance(y);
                                        if(cet.SequenceNumber == hlt.SequenceNumber)
                                        {
                                            ext = cet;
                                            sb.AppendLine(string.Format("Using instance {1} of {0}", ext.TableName, y + 1));
                                            break;
                                        }
                                    }
                                }
                                break;
                            }
                        }
                    }

                    if (ext != null && hlt.Fields != null && hlt.Fields.Length > 0)
                    {
                        for (int x = 0; x < hlt.Fields.Length; x++)
                        {
                            if (hlt.Fields[x].EditStatus != "HLModifiedEntry")
                                continue;

                            CExtField ef = ext.GetField(hlt.Fields[x].FieldName);
                            if (!hlt.Fields[x].Equals(ef))
                            {
                                if (create)
                                {
                                    ext = c.NewExtTableRow(ext.TableName);
                                    ef = ext.GetField(hlt.Fields[x].FieldName);
                                    create = false;
                                    sb.AppendLine(string.Format("Create new instance of {0}.  Instance count = {1}", ext.TableName, ext.InstancesCount));
                                }
                                else
                                {
                                    sb.AppendLine(string.Format("Update instance of {0}.  Instance count = {1}", ext.TableName, ext.InstancesCount));
                                }
                                sb.AppendLine(string.Format("Update {0}.{1} from '{2}' to '{3}'", ext.TableName, ef.FieldName, ef.FieldValue, hlt.Fields[x].GetValue()));
                                ef.FieldValue = hlt.Fields[x].GetValue();
                            }
                        }
                    }
                    else
                    {
                        sb.AppendLine(string.Format("Could not find {0}", ext.TableName));
                    }
                }
            }
            else
            {
                sb.AppendLine(string.Format("No extender tables"));
            }

            c.Save();

            result = new HLImisContact();
            result.AuditInfo = sb.ToString();
            result.Load(c);

            return result;
        }
        catch (Exception excp)
        {
            result = new HLImisContact();
            if (c != null)
                result.Load(c);
            result.AuditInfo = sb.ToString();
            result.ExceptionMessage = excp.Message + excp.StackTrace;
            return result;
        }
    }

    [WebMethod(Description = "Sets the contact's password.", EnableSession = false)]
    public string SetPassword(string securityKey, string webLogin, string password)
    {
        try
        {
            CContactUser user = CContactUser.LoginByWebLogin(webLogin);
            user.SetPassword(Decrypt(password));
        }
        catch (Exception excp)
        {
            return excp.Message;
        }
        return "OK";
    }

    [WebMethod(Description = "Updates the contact's password.", EnableSession = false)]
    public string UpdatePassword(string securityKey, string webLogin, string currentPassword, string newPassword)
    {
        try
        {
            string cp = Decrypt(currentPassword);
            string np = Decrypt(newPassword);

            CContactUser user = CContactUser.LoginByWebLogin(webLogin, cp);
            user.ChangePassword(cp, np);
        }
        catch(Exception excp)
        {
            return excp.Message;
        }
        return "OK";
    }

    private static void ValidateSecurityKey(string securityKey)
    {
        if (string.IsNullOrEmpty(securityKey) || string.Compare(securityKey, ConfigurationManager.AppSettings["SecurityKey"]) != 0)
            throw new AccessViolationException("Unauthorized Access.");
    }

    public class HLImisAddress
    {
        public string Address1;
        public string Address2;
        public string Address3;
        public int AddressNumber;
        public string AddressPurpose;
        public string City;
        public string Country;
        public string County;
        public string EditStatus;
        public string EmailAddress;
        public HLImisErrors Errors;
        public string Fax;
        public string InstituteName;
        public bool IsPreferredBill;
        public bool IsPreferredMail;
        public bool IsPreferredShip;
        public string Phone;
        public string PostalCode;
        public string StateProvince;
        public string TollFree;

        public void Load(CAddress addr)
        {
            Address1 = addr.Address1;
            Address2 = addr.Address2;
            Address3 = addr.Address3;
            AddressNumber = addr.AddressNumber;
            AddressPurpose = addr.AddressPurpose;
            City = addr.City;
            Country = addr.Country;
            County = addr.County;
            EditStatus = addr.EditStatus.ToString();
            EmailAddress = addr.EmailAddress;
            Fax = addr.Fax;
            InstituteName = addr.InstituteName;
            IsPreferredBill = addr.IsPreferredBill;
            IsPreferredMail = addr.IsPreferredMail;
            IsPreferredShip = addr.IsPreferredShip;
            Phone = addr.Phone;
            PostalCode = addr.PostalCode;
            StateProvince = addr.StateProvince;
            TollFree = addr.TollFree;

            if (addr.ErrorsCount > 0)
            {
                Errors = new HLImisErrors();
                Errors.Load(addr.Errors);
            }
        }
    }

    public class HLImisContact
    {
        public HLImisAddress[] Addresses;
        public string AuditInfo;
        public string BirthDate;
        public string Chapter;
        public string ContactId;
        public string ContactStatusCode;
        public string CustomerTypeCode;
        public string Designation;
        public string EmailAddress;
        public HLImisErrors Errors;
        public string ExceptionMessage = string.Empty;
        public bool ExcludeFromDirectory;
        public bool ExcludeFromMailings;
        public HLImisExtTable[] ExtTables;
        public string Fax;
        public string FirstName;
        public string Gender;
        public string HomePhone;
        public string Informal;
        public string InstituteContactId;
        public string InstituteCustomerTypeCode;
        public string InstituteName;
        public bool IsInstitute;
        public bool IsMember;
        public string JoinDate;
        public string LastName;
        public string MemberStatusCode;
        public string MemberStatusDate;
        public string MiddleName;
        public string Prefix;
        public string Suffix;
        public string Title;
        public string TollFreePhone;
        public string WebsiteAddress;
        public string WorkPhone;

        

        public void Load(CContact c)
        {
            StringBuilder sb = new StringBuilder();

            BirthDate = c.BirthDate.ToString("yyyy-MM-dd HH:mm:ss");
            Chapter = c.Chapter;
            ContactId = c.ContactId;
            ContactStatusCode = c.ContactStatusCode;
            CustomerTypeCode = c.CustomerTypeCode;
            Designation = c.Designation;
            EmailAddress = c.EmailAddress;
            ExcludeFromDirectory = c.ExcludeFromDirectory;
            ExcludeFromMailings = c.ExcludeFromMailings;
            Fax = c.Fax;
            FirstName = c.FirstName;
            Gender = c.Gender.ToString(); // Female, Male or Unspecified
            HomePhone = c.HomePhone;
            Informal = c.Informal;
            InstituteContactId = c.InstituteContactId;
            InstituteCustomerTypeCode = c.InstituteCustomerTypeCode;
            InstituteName = c.InstituteName;
            IsInstitute = c.IsInstitute;
            IsMember = c.IsMember;
            JoinDate = c.JoinDate.ToString("yyyy-MM-dd HH:mm:ss");
            LastName = c.LastName;
            MemberStatusCode = c.MemberStatusCode;
            MemberStatusDate = c.MemberStatusDate.ToString("yyyy-MM-dd HH:mm:ss");
            MiddleName = c.MiddleName;
            Prefix = c.Prefix;
            Suffix = c.Suffix;
            Title = c.Title;
            TollFreePhone = c.TollFreePhone;
            WebsiteAddress = c.WebsiteAddress;
            WorkPhone = c.WorkPhone;

            if (c.AddressCount > 0)
            {
                CAddress[] addrs = c.Addresses;
                Addresses = new HLImisAddress[c.AddressCount];
                for (int x = 0; x < addrs.Length; x++)
                {
                    Addresses[x] = new HLImisAddress();
                    Addresses[x].Load(addrs[x]);
                }
            }

            if (c.ExtTableCount > 0)
            {
                CExtTable[] tables = c.ExtTables;
                HLImisExtTable het;
                List<HLImisExtTable> list = new List<HLImisExtTable>();
                
                for (int x = 0; x < tables.Length; x++)
                {
                    if(tables[x].InstancesCount == 0)
                    {
                        het = new HLImisExtTable();
                        het.Load(tables[x], false);
                        list.Add(het);
                        sb.AppendLine(string.Format("Load {0} Non-Instance", tables[x].TableName));
                    }
                    else
                    {
                        for(int y = 0; y < tables[x].InstancesCount; y++)
                        {
                            het = new HLImisExtTable();
                            het.Load(tables[x].GetInstance(y), false);
                            list.Add(het);
                            sb.AppendLine(string.Format("Load {0} Instance {1}", tables[x].TableName, y + 1));
                        }
                    }
                    
                }
                ExtTables = list.ToArray();
            }

            if (c.ErrorsCount > 0)
            {
                CErrors errors = c.Errors;
                Errors = new HLImisErrors();
                Errors.Load(c.Errors);
            }

            if (string.IsNullOrEmpty(AuditInfo))
                AuditInfo = sb.ToString();
            else
                AuditInfo += (Environment.NewLine + sb.ToString());
        }
    }

    public class HLImisErrors
    {
        public int ErrorCount;
        public HLImisError[] Errors;
        public string PrimaryErrorMessage;
        public int WarningCount;

        public void Load(CErrors errors)
        {
            if (errors == null)
                return;

            Errors = new HLImisError[errors.ErrorCount];
            ErrorCount = errors.ErrorCount;
            PrimaryErrorMessage = errors.PrimaryErrorMessage;
            WarningCount = errors.WarningCount;
            for (int x = 0; x < errors.ErrorCount; x++)
            {
                CError ce = errors.GetError(x);
                Errors[x] = new HLImisError();
                Errors[x].Load(ce);
            }
        }
    }

    public class HLImisError
    {
        public string ErrorCategory;
        public string ErrorNumber;
        public string Location;
        public string Message;
        
        public void Load(CError error)
        {
            if (error == null)
                return;

            ErrorCategory = error.Category.ToString();
            ErrorNumber = error.Number.ToString();
            Location = error.Location;
            Message = error.Message;
        }
    }

    public class HLImisExtTable
    {
        public string Application;
        public string Description;
        public string EditStatus;
        public string EditTypes;
        public HLImisErrors Errors;
        public HLImisExtField[] Fields;
        public int InstancesCount;
        public bool IsEntryRequired;
        public bool IsExternal;
        public bool IsMultiInstance;
        public string LinkViaColumnName;
        public int SequenceNumber = -1;
        public string TableName;

        public void Load(CExtTable source, bool includeLookupValues)
        {
            Application = source.Application;
            Description = source.Description;
            EditStatus = source.EditStatus.ToString();
            EditTypes = source.EditTypes;
            IsEntryRequired = source.IsEntryRequired;
            IsExternal = source.IsExternal;
            IsMultiInstance = source.IsMultiInstance;
            LinkViaColumnName = source.LinkViaColumnName;
            SequenceNumber = source.SequenceNumber;
            TableName = source.TableName;


            if (source.ErrorsCount > 0)
            {
                CErrors errors = source.Errors;
                Errors = new HLImisErrors();
                Errors.Load(source.Errors);
            }

            CExtField[] f = source.Fields;
            if (f != null && f.Length > 0)
            {
                Fields = new HLImisExtField[f.Length];
                for (int x = 0; x < f.Length; x++)
                {
                    Fields[x] = new HLImisExtField();
                    Fields[x].Load(f[x], includeLookupValues);
                }
            }
        }
    }

    public class HLImisExtField
    {
        public string DataType;
        public int DecimalPositions;
        public string DefaultValue;
        public string EditStatus;
        public HLImisErrors Errors;
        public int FieldLength;
        public string FieldName;
        public string FieldValue;
        public string FormatMask;
        public HLImisKeyValuePair[] LookupValues;
        public bool MultiSelect;
        public string Prompt;
        public int Sequence;
        public string ValidationTable;

        public bool Equals(CExtField ef)
        {
            if (ef == null)
                return false;

            if (ef.FieldValue == null)
                return string.IsNullOrEmpty(FieldValue);

            if (ef.FieldValue is string)
                return string.Compare(ef.FieldValue as string, FieldValue) == 0;

            try
            {
                Type dt = ef.FieldValue.GetType();
                if (dt == typeof(bool))
                {
                    bool b;
                    if (string.IsNullOrEmpty(FieldValue) || !bool.TryParse(FieldValue, out b))
                        b = false;
                    return b == (bool)ef.FieldValue;
                }
                else if (dt == typeof(DateTime))
                {
                    DateTime t;
                    if (string.IsNullOrEmpty(FieldValue) || !DateTime.TryParse(FieldValue, out t))
                        t = DateTime.MinValue;
                    return t == (DateTime)ef.FieldValue;
                }
                else if (dt == typeof(decimal))
                {
                    decimal d;
                    if (string.IsNullOrEmpty(FieldValue) || !decimal.TryParse(FieldValue, out d))
                        d = 0;
                    return d == (decimal)ef.FieldValue;
                }
                else if (dt == typeof(double))
                {
                    double db;
                    if (string.IsNullOrEmpty(FieldValue) || !double.TryParse(FieldValue, out db))
                        db = 0;
                    return db == (double)ef.FieldValue;
                }
                else if (dt == typeof(int))
                {
                    int i;
                    if (string.IsNullOrEmpty(FieldValue) || !int.TryParse(FieldValue, out i))
                        i = 0;
                    return i == (int)ef.FieldValue;
                }

                return string.Compare(ef.FieldValue.ToString(), FieldValue) == 0;
            }
            catch(Exception excp)
            {
                throw new Exception(string.Format("{0} {1} {2}", ef.FieldValue.GetType().Name, ef.DataType, excp.Message));
            }

        }

        public object GetValue()
        {
            switch (DataType)
            {
                case "Char":
                    return FieldValue;
                case "Int":
                    int i;
                    if (string.IsNullOrEmpty(FieldValue) || !int.TryParse(FieldValue, out i))
                        i = 0;
                    return i;
                case "Number":
                case "Money":
                    decimal d;
                    if (string.IsNullOrEmpty(FieldValue) || !decimal.TryParse(FieldValue, out d))
                        d = 0;
                    return d;
                case "CheckBox":
                    bool b;
                    if (string.IsNullOrEmpty(FieldValue) || !bool.TryParse(FieldValue, out b))
                        b = false;
                    return b;
                case "Date":
                case "Time":
                case "DateTime":
                    DateTime t;
                    if (string.IsNullOrEmpty(FieldValue) || !DateTime.TryParse(FieldValue, out t))
                        t = DateTime.MinValue;
                    return t;
            }
            return FieldValue;
        }

        public void Load(CExtField source, bool includeLookupValues)
        {
            DataType = source.DataType;
            DecimalPositions = source.DecimalPositions;
            DefaultValue = source.DefaultValue;
            EditStatus = source.EditStatus.ToString();
            FieldLength = source.FieldLength;
            FieldName = source.FieldName;
            if (source.FieldValue != null)
                FieldValue = source.FieldValue.ToString();
            FormatMask = source.FormatMask;
            if (includeLookupValues && !string.IsNullOrEmpty(source.ValidationTable))
                LookupValues = GetLookup(source.ValidationTable);
            MultiSelect = source.MultiSelect;
            Prompt = source.Prompt;
            Sequence = source.Sequence;
            ValidationTable = source.ValidationTable;

            if (source.ErrorsCount > 0)
            {
                CErrors errors = source.Errors;
                Errors = new HLImisErrors();
                Errors.Load(source.Errors);
            }
        }
    }

    public class HLImisKeyValuePair
    {
        public HLImisKeyValuePair()
        {
        }

        public HLImisKeyValuePair(string key, string value)
        {
            Key = key;
            Value = value;
        }

        public string Key;
        public string Value;
    }

    public class HLImisReference
    {
        public string Address2Prompt;
        public string Address3Prompt;
        public string[] AddressPurposes;
        public string ChapterPrompt;
        public string[] CountryNames;
        public string CountyPrompt;
        public HLImisExtTable[] ExtTables;
        public string InstitutePrompt;
        public string MainAddressPrompt;
        public string[] Prefixes;
        public string StatePrompt;
        public string[] StatesProvinces;
        public string[] Suffixes;
        public string TollFreePrompt;

        public string ExceptionMessage = string.Empty;

    }
}
