﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Web.Services;
using System.Data;
using System.Data.SqlClient;
using Asi.iBO;
using Asi.iBO.ContactManagement;
using AccessViolationException = System.AccessViolationException;
using DataServer = HigherLogic.Data.DataServer;

#region Structures
/// <summary>
/// Represents a security token for use with Authorize and Authenticate
/// </summary>
public struct AuthToken
{
    public string MemberKey;
    public DateTime ValidUntil;

    public AuthToken(string memberKey, DateTime validUntil)
    {
        MemberKey = memberKey;
        ValidUntil = validUntil;
    }
}

/// <summary>
/// Represents a subset of memberinfo
/// </summary>
public struct MemberInfo
{
    public string ID;
    public string Email; 
    public string Username;

    public MemberInfo(string id, string email, string username)
    {
        ID = id;
        Email = email;
        Username = username;
    }
}

/// <summary>
/// Represents a Group Type
/// </summary>
public struct GroupType
{
    public string GroupKey;
    public string GroupName;

    public GroupType(string groupKey, string groupName)
    {
        GroupKey = groupKey;
        GroupName = groupName;
    }
}

/// <summary>
/// Represents a group that a member belongs to
/// </summary>
public struct MemberGroup
{
    public string GroupKey;
    public string GroupName;
    public string GroupType;
    public string Role;
    public DateTime SinceDate;
    public DateTime StartDate;
    public DateTime EndDate;

    public MemberGroup(string groupKey, string groupName, string groupType, string role, DateTime sinceDate, DateTime startDate, DateTime endDate)
    {
        GroupKey = groupKey;
        GroupName = groupName;
        GroupType = groupType;
        Role = role;
        SinceDate = sinceDate;
        StartDate = startDate;
        EndDate = endDate;
    }
}

/// <summary>
/// Represents a single member's membership in a group
/// </summary>
public struct GroupMember
{
    public string MemberKey;
    public string Role;
    public DateTime SinceDate;
    public DateTime StartDate;
    public DateTime EndDate;

    public GroupMember(string memberKey, string role, DateTime sinceDate, DateTime startDate, DateTime endDate)
    {
        MemberKey = memberKey;
        Role = role;
        SinceDate = sinceDate;
        StartDate = startDate;
        EndDate = endDate;
    }
}

/// <summary>
/// Represents a single Meeting / Calendar Event
/// </summary>
public struct Meeting
{
    public string LegacyEventKey;
    public string EventTitle;
    public string EventTypeKey;
    public string EventDescription;
    public DateTime StartDateTime;
    public DateTime EndDateTime;
    public string Address1;
    public string Address2;
    public string Address3;
    public string City;
    public string StateProvinceCode;
    public string PostalCode;
    public string CountryCode;
    public DateTime EarlyRegistrationDate;
    public DateTime RegularRegistrationDate;
    public DateTime LastRegistrationDate;
    public string DetailURL;
    public string Status;
    public MeetingFunction[] Functions;

    public Meeting(string legacyEventKey, string eventTitle, string eventTypeKey, string eventDescription, DateTime startDateTime, DateTime endDateTime, string address1,
                    string address2,string address3,string city,string stateProvinceCode,string postalCode,string countryCode,DateTime earlyRegistrationDate,
                    DateTime regularRegistrationDate,DateTime lastRegistrationDate,string detailURL,string status, MeetingFunction[] functions)
    {
        LegacyEventKey = legacyEventKey;
        EventTitle = eventTitle;
        EventTypeKey = eventTypeKey;
        EventDescription = eventDescription;
        StartDateTime = startDateTime;
        EndDateTime = endDateTime;
        Address1 = address1;
        Address2 = address2;
        Address3 = address3;
        City = city;
        StateProvinceCode = stateProvinceCode;
        PostalCode = postalCode;
        CountryCode = countryCode;
        EarlyRegistrationDate = earlyRegistrationDate;
        RegularRegistrationDate = regularRegistrationDate;
        LastRegistrationDate = lastRegistrationDate;
        DetailURL = detailURL;
        Status = status;
        Functions = functions;
    }
}
public struct MeetingFunction
{
    public string LegacyOptionKey;
    public string FunctionTitle;
    public string FunctionTypeKey;
    public string FunctionDescription;
    public DateTime StartDateTime;
    public DateTime EndDateTime;
    public string CECredits;
    public string SessionCode;
    public string Status;

    public MeetingFunction(string legacyOptionKey,string functionTitle,string functionTypeKey,string functionDescription,DateTime startDateTime,
                         DateTime endDateTime,string cECredits,string sessionCode,string status)
    {
        LegacyOptionKey = legacyOptionKey;
        FunctionTitle = functionTitle;
        FunctionTypeKey = functionTypeKey;
        FunctionDescription = functionDescription;
        StartDateTime = startDateTime;
        EndDateTime = endDateTime;
        CECredits = cECredits;
        SessionCode = sessionCode;
        Status = status;
    }
}

/// <summary>
/// Gen_Table Row
/// </summary>
public class LookupCode
{
    string _code;
    string _description;

    public LookupCode()
    {
    }

    public LookupCode(string code, string description)
    {
        Code = code;
        Description = description;
    }

    public string Code
    {
        get { return _code; }
        set { _code = value; }
    }

    public string Description
    {
        get { return _description; }
        set { _description = value; }
    }
}

/// <summary>
/// Gen_Table Rows
/// </summary>
public class LookupData
{
    List<LookupCode> _codes;

    string _name;

    public LookupData()
    {
    }

    public LookupData(string name)
    {
        Name = name;
    }

    public void AddCode(string code, string description)
    {
        Codes.Add(new LookupCode(code, description));
    }

    public List<LookupCode> Codes
    {
        get
        {
            if (_codes == null)
                _codes = new List<LookupCode>();
            return _codes;
        }
        set { _codes = value; }
    }

    public string Name
    {
        get { return _name; }
        set { _name = value; }
    }
}
/// <summary>
/// Represents a single member's profile details
/// </summary>
public struct MemberAddress
{
    public string AddressType;
    public string Address1;
    public string Address2;
    public string Address3;
    public string City;
    public string StateProvince;
    public string PostalCode;
    public string CountryCode;
    public string Phone1;
    public string Phone1Type;
    public string Phone2;
    public string Phone2Type;
    public string Phone3;
    public string Phone3Type;
    public string Phone4;
    public string Phone4Type;
    public string EmailAddress;
    public bool IsPreferredMail;
    public bool IsPreferredBill;
    public bool IsPreferredList;
    public bool IsListCityState;
    public string LegacyAddressKey;

    public MemberAddress(string addressType, string address1, string address2, string address3, string city, string stateProvince, string postalCode, string countryCode, string phone1, string phone1Type, string phone2, string phone2Type, string phone3, string phone3Type, string phone4, string phone4Type, string emailAddress, bool isPreferredMail, bool isPreferredBill, bool isPreferredList, bool isListCityState, string legacyAddressKey)
    {
        AddressType = addressType;
        Address1 = address1;
        Address2 = address2;
        Address3 = address3;
        City = city;
        StateProvince = stateProvince;
        PostalCode = postalCode;
        CountryCode = countryCode;
        Phone1 = phone1;
        Phone1Type = phone1Type;
        Phone2 = phone2;
        Phone2Type = phone2Type;
        Phone3 = phone3;
        Phone3Type = phone3Type;
        Phone4 = phone4;
        Phone4Type = phone4Type;
        EmailAddress = emailAddress;
        IsPreferredMail = isPreferredMail;
        IsPreferredBill = isPreferredBill;
        IsPreferredList = isPreferredList;
        IsListCityState = isListCityState;
        LegacyAddressKey = legacyAddressKey;
    }
}

/// <summary>
/// Represents a single member's profile details
/// </summary>
public struct MemberDetails
{
    public string MemberKey;
    public string Prefix;
    public string FirstName;
    public string LastName;
    public string Suffix;
    public string Designation;
    public string InformalName;
    public string CompanyName;
    public string Title;
    public string Email1;
    public string Email2;
    public string Phone1;
    public string Phone1Type;
    public string Phone2;
    public string Phone2Type;
    public string Phone3;
    public string Phone3Type;
    public string Phone4;
    public string Phone4Type;
    public string Address1;
    public string Address2;
    public string Address3;
    public string City;
    public string StateProvince;
    public string PostalCode;
    public string CountryCode;
    public string WebsiteUrl;
    public string PictureUrl;
    public string Status;
    public bool IsCompany;
    public string CompanyID;
    public bool DoNotList;
	public bool DoNotEmail;
    public string Custom01;
    public string Custom02;
    public string Custom03;
    public string Custom04;
    public string Custom05;
    public string Custom06;
    public string Custom07;
    public string Custom08;
    public string Custom09;
    public string Custom10;
    public MemberAddress[] Addresses;

    //    public DemographicField[] Demographics;
    //    public Activity[] Activities;
    //    public Subscription[] Subscriptions;

    public MemberDetails(string memberKey, string prefix, string firstName, string lastName, string suffix, string designation, string informalName, string companyName, string title, string email1, string email2, string phone1, string phone1Type, string phone2, string phone2Type, string phone3, string phone3Type, string phone4, string phone4Type, string address1, string address2, string address3, string city, string stateProvince, string postalCode, string countryCode, string websiteUrl, string pictureUrl, string status, bool isCompany, string companyID, bool doNotList, bool doNotEmail, string custom01, string custom02, string custom03, string custom04, string custom05, string custom06, string custom07, string custom08, string custom09, string custom10)
    {
        MemberKey = memberKey;
        Prefix = prefix;
        FirstName = firstName;
        LastName = lastName;
        Suffix = suffix;
        Designation = designation;
        InformalName = informalName;
        CompanyName = companyName;
        Title = title;
        Email1 = email1;
        Email2 = email2;
        Phone1 = phone1;
        Phone1Type = phone1Type;
        Phone2 = phone2;
        Phone2Type = phone2Type;
        Phone3 = phone3;
        Phone3Type = phone3Type;
        Phone4 = phone4;
        Phone4Type = phone4Type;
        Address1 = address1;
        Address2 = address2;
        Address3 = address3;
        City = city;
        StateProvince = stateProvince;
        PostalCode = postalCode;
        CountryCode = countryCode;
        WebsiteUrl = websiteUrl;
        PictureUrl = pictureUrl;
        Status = status;
        IsCompany = isCompany;
        CompanyID = companyID;
        DoNotList = doNotList;
		DoNotEmail = doNotEmail;
        Custom01 = custom01;
        Custom02 = custom02;
        Custom03 = custom03;
        Custom04 = custom04;
        Custom05 = custom05;
        Custom06 = custom06;
        Custom07 = custom07;
        Custom08 = custom08;
        Custom09 = custom09;
        Custom10 = custom10;
        Addresses = new MemberAddress[0];
    }
}

public struct GroupCount
{
    public string LegacyGroupKey;
    public int Count;

    public GroupCount(string legacyGroupKey, int count)
    {
        LegacyGroupKey = legacyGroupKey;
        Count = count;
    }
}

public struct ChangedMember
{
    public string LegacyContactKey;
    public DateTime ChangeDateTime;

    public ChangedMember(string legacyContactKey, DateTime changeDateTime)
    {
        LegacyContactKey = legacyContactKey;
        ChangeDateTime = changeDateTime;
    }
}

public struct DemographicField
{
    public string InstanceKey;
    public string ClassName;
    public string PropertyName;
    public string PropertyValue;

    public DemographicField(string instanceKey, string className, string propertyName, string propertyValue)
    {
        InstanceKey = instanceKey;
        ClassName = className;
        PropertyName = propertyName;
        PropertyValue = propertyValue;
    }
}

public struct Activity
{
    public string ActivityKey;
    public string ActivityType;
    public string ActivityCode;
    public DateTime ActivityDate;
    public string Description;
    public string Details;
    public string Quantity;
    public string Amount;

    public Activity(string activityKey, string activityType, string activityCode, DateTime activityDate, string description, string details, string quantity, string amount)
    {
        ActivityKey = activityKey;
        ActivityType = activityType;
        ActivityCode = activityCode;
        ActivityDate = activityDate;
        Description = description;
        Details = details;
        Quantity = quantity;
        Amount = amount;
    }
}

public struct Subscription
{
    public string SubscriptionKey;
    public string SubscriptionType;
    public string Description;
    public DateTime SinceDate;
    public DateTime StartDate;
    public DateTime EndDate;

    public Subscription(string subscriptionKey, string subscriptionType, string description, DateTime sinceDate, DateTime startDate, DateTime endDate)
    {
        SubscriptionKey = subscriptionKey;
        SubscriptionType = subscriptionType;
        Description = description;
        SinceDate = sinceDate;
        StartDate = startDate;
        EndDate = endDate;
    }
}

/// <summary>
/// User Defined Demographic Table
/// </summary>
public class UDData
{
    List<UDItem> _items;

    string _tableName;

    public UDData()
    {
    }

    public UDData(string tableName)
    {
        TableName = tableName;
    }

    public UDItem AddItem(string id)
    {
        UDItem item = new UDItem(id);
        Items.Add(item);
        return item;
    }

    public UDItem AddItem(string id, int seqn)
    {
        UDItem item = new UDItem(id, seqn);
        Items.Add(item);
        return item;
    }

    public List<UDItem> Items
    {
        get
        {
            if (_items == null)
                _items = new List<UDItem>();
            return _items;
        }
        set { _items = value; }
    }

    public string TableName
    {
        get { return _tableName; }
        set { _tableName = value; }
    }
}

/// <summary>
/// User Defined Demographic Table Row
/// </summary>
public class UDItem
{
    List<UDValue> _values;

    string _id;
    int _seqn;

    public UDItem()
    {
    }

    public UDItem(string id)
    {
        ID = id;
    }

    public UDItem(string id, int seqn)
    {
        ID = id;
        Seqn = seqn;
    }

    public void AddColumn(string fieldName, object value)
    {
        Values.Add(new UDValue(fieldName, value));
    }

    public string ID
    {
        get { return _id; }
        set { _id = value; }
    }

    public int Seqn
    {
        get { return _seqn; }
        set { _seqn = value; }
    }

    public List<UDValue> Values
    {
        get
        {
            if (_values == null)
                _values = new List<UDValue>();
            return _values;
        }
        set { _values = value; }
    }
}

/// <summary>
/// UD_Table
/// </summary>
public class UDItemMeta
{
    List<UDValueMeta> _valueMeta;

    bool _allowMultiple;
    string _description;
    string _tableName;

    public UDItemMeta()
    {
    }

    public UDItemMeta(string tableName, string description, bool allowMultiple)
    {
        TableName = tableName;
        Description = description;
        AllowMultiple = allowMultiple;
    }

    public void AddValueMeta(string fieldName, string fieldType, int length, int scale, string prompt, string lookupName, bool multiSelect)
    {
        ValueMeta.Add(new UDValueMeta(fieldName, fieldType, length, scale, prompt, lookupName, multiSelect));
    }

    public bool AllowMultiple
    {
        get { return _allowMultiple; }
        set { _allowMultiple = value; }
    }

    public string Description
    {
        get { return _description; }
        set { _description = value; }
    }

    public string TableName
    {
        get { return _tableName; }
        set { _tableName = value; }
    }

    public List<UDValueMeta> ValueMeta
    {
        get
        {
            if (_valueMeta == null)
                _valueMeta = new List<UDValueMeta>();
            return _valueMeta;
        }
        set { _valueMeta = value; }
    }
}

/// <summary>
/// User Defined Demographic Table Column
/// </summary>
public class UDValue
{
    string _fieldName;
    bool _isNull;
    string _value;

    public UDValue()
    {
    }

    public UDValue(string fieldName, object value)
    {
        FieldName = fieldName;
        if (value == null || value == DBNull.Value)
        {
            IsNull = true;
        }
        else
        {
            Type type = value.GetType();
            if (type == typeof(string))
                Value = (string)value;
            else if (type == typeof(bool))
                Value = ((bool)value ? bool.TrueString : bool.FalseString);
            else if (type == typeof(DateTime))
                Value = ((DateTime)value).ToString("U", System.Globalization.CultureInfo.InvariantCulture);
            else if (type == typeof(int))
                Value = ((int)value).ToString(System.Globalization.CultureInfo.InvariantCulture);
            else if (type == typeof(decimal))
                Value = ((decimal)value).ToString(System.Globalization.CultureInfo.InvariantCulture);
            else
                Value = value.ToString();
        }
    }

    public string FieldName
    {
        get { return _fieldName; }
        set { _fieldName = value; }
    }

    public bool IsNull
    {
        get { return _isNull; }
        set { _isNull = value; }
    }

    public string Value
    {
        get { return _value; }
        set { _value = value; }
    }

    public bool ValueAsBool
    {
        get { return Value == bool.TrueString; }
    }

    public DateTime ValueAsDateTime
    {
        get { return DateTime.ParseExact(Value, "U", System.Globalization.CultureInfo.InvariantCulture); }
    }

    public decimal ValueAsDecimal
    {
        get { return decimal.Parse(Value, System.Globalization.CultureInfo.InvariantCulture); }
    }

    public int ValueAsInt
    {
        get { return int.Parse(Value, System.Globalization.CultureInfo.InvariantCulture); }
    }

    public string ValueAsString
    {
        get { return Value; }
    }
}

/// <summary>
/// UD_Field
/// </summary>
public class UDValueMeta
{
    string _fieldName;
    string _fieldType;
    int _length;
    string _lookupName;
    bool _multiSelect;
    string _prompt;
    int _scale;

    public UDValueMeta()
    {
    }

    public UDValueMeta(string fieldName, string fieldType, int length, int scale, string prompt, string looupName, bool multiSelect)
    {
        FieldName = fieldName;
        FieldType = fieldType;
        Length = length;
        Scale = scale;
        Prompt = prompt;
        LookupName = looupName;
        MultiSelect = multiSelect;
    }

    public string FieldName
    {
        get { return _fieldName; }
        set { _fieldName = value; }
    }

    public string FieldType
    {
        get { return _fieldType; }
        set { _fieldType = value; }
    }

    public int Length
    {
        get { return _length; }
        set { _length = value; }
    }

    public string LookupName
    {
        get { return _lookupName; }
        set { _lookupName = value; }
    }

    public bool MultiSelect
    {
        get { return _multiSelect; }
        set { _multiSelect = value; }
    }

    public string Prompt
    {
        get { return _prompt; }
        set { _prompt = value; }
    }

    public int Scale
    {
        get { return _scale; }
        set { _scale = value; }
    }
}

#endregion

/// <summary>
/// Summary description for HigherLogicWebService
/// </summary>
[WebService(Namespace = "http://higherlogic.com/")]
[WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
// To allow this Web Service to be called from script, using ASP.NET AJAX, uncomment the following line. 
// [System.Web.Script.Services.ScriptService]
public class HigherLogicWebService : WebService
{
    private static IiMISUser _iMISUser = null;
    private static Dictionary<string, List<string>> _DemographicFields = null;
    private static string[] _ReadActivityTypes = null;
    private static string[] _ReadSubscriptionTypes = null;
    private static Dictionary<Guid, AuthToken> _AuthTokens = null;

    public HigherLogicWebService()
    {
        if (_AuthTokens == null)
            _AuthTokens = new Dictionary<Guid, AuthToken>();
    }

    private static bool CheckSecurity(string securityKey)
    {
        return securityKey == ConfigurationManager.AppSettings["SecurityKey"];
    }

    [WebMethod(Description = "Just returns 0 to see how long a round trip takes.", EnableSession = false)]
    public long Ping()
    {
        return 0L;
    }

    [WebMethod(Description = "Authorize a MemberKey and get back a token.", EnableSession = false)]
    public Guid Authorize(string securityKey, string memberKey)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");
        Guid guid = Guid.NewGuid();

        AuthToken authToken = new AuthToken(memberKey, DateTime.Now.AddMinutes(30));
        _AuthTokens.Add(guid, authToken);
        return guid;
    }

    [WebMethod(Description = "Authenticate a Guid and get back a MemberID.", EnableSession = false)]
    public String Authenticate(string securityKey, Guid guid)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        List<Guid> oldGuids = new List<Guid>();
        foreach (Guid authGuid in _AuthTokens.Keys)
        {
            AuthToken authToken = _AuthTokens[authGuid];
            if (authToken.ValidUntil.CompareTo(DateTime.Now) < 0)
                oldGuids.Add(authGuid);
        }
        foreach (Guid oldGuid in oldGuids)
            _AuthTokens.Remove(oldGuid);

        if (_AuthTokens.ContainsKey(guid))
            return _AuthTokens[guid].MemberKey;

        return "NotAuthenticated";
    }
    [WebMethod(Description = "Decrypt UserId to get back a MemberID.", EnableSession = false)]
    public String AuthenticateUserID(string securityKey, string userid)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        if (!iboAdmin.IsSystemInitialized)
            iboAdmin.InitializeSystem();


        try
        {
            Asi.iBO.CCrypto crypt = new CCrypto();
            string myID = crypt.Decrypt(userid);
            return myID;

        }
        catch { }

        return string.Empty;
    }
    [WebMethod(Description = "Authenticate a user by userid and password stored in the AMS.", EnableSession = false)]
    public String AuthenticateUser(string securityKey, string userName, string password)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        if (!iboAdmin.IsSystemInitialized)
            iboAdmin.InitializeSystem();

        try
        {
            CContactUser user = CContactUser.LoginByWebLogin(userName, password);
            return user.ContactId;
        }
        catch {}

        return string.Empty;
    }


    [WebMethod(Description = "Returns the MemberInfo given an emailAddress", EnableSession = false)]
    public MemberInfo[] GetIDfromEmailAddress(string securityKey, string emailAddress)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        List<MemberInfo> list = new List<MemberInfo>();
        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[1];
            param[0] = new SqlParameter("@EmailAddress", SqlDbType.VarChar);
            param[0].Value = emailAddress;

            using (SqlDataReader reader = server.ExecuteReader(CommandType.Text, "select top 10 n.ID, n.EMAIL, s.WEB_LOGIN from Name n, Name_Security s where n.ID = s.ContactID and n.EMAIL like @EmailAddress", param))
            {
                while (reader.Read())
                {
                    list.Add(new MemberInfo((string)reader["ID"], (string)reader["EMAIL"], (string)reader["WEB_LOGIN"]));
                }
                reader.Close();
            }
        }
        return list.ToArray();
    }

    [WebMethod(Description = "Gets a list Code/Description pairs for the specified Gen_Table name.", EnableSession = false)]
    public LookupData GetLookupData(string securityKey, string name)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        LookupData data = new LookupData(name);
        SqlParameter[] param;
        string sql;

        sql = @"select UPPER_CODE,DESCRIPTION
                     from  Gen_Tables
                     where TABLE_NAME = @TableName
                     order by UPPER_CODE";
        param = new SqlParameter[1];
        param[0] = new SqlParameter("@TableName", SqlDbType.VarChar);
        param[0].Value = name;

        using (DataServer server = new DataServer())
        {
            using (SqlDataReader reader = server.ExecuteReader(CommandType.Text, sql, param))
            {
                while (reader.Read())
                    data.AddCode((string)reader["UPPER_CODE"], (string)reader["DESCRIPTION"]);
                reader.Close();
            }
        }

        return data;
    }

    [WebMethod(Description = "Gets the list of security groups with their associated counts.", EnableSession = false)]
    public GroupCount[] GetSecurityGroups(string securityKey)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        List<GroupCount> list = new List<GroupCount>();

        using (DataServer server = new DataServer())
        {
            using (SqlDataReader reader = server.ExecuteReader(CommandType.StoredProcedure, "hlGetSecurityGroups"))
            {
                while (reader.Read())
                {
                    list.Add(new GroupCount((string)reader["GroupKey"], (int)reader["Count"]));
                }
                reader.Close();
            }
        }
        return list.ToArray();
    }

    [WebMethod(Description = "Gets the list of members whose data has changed since the date provided.", EnableSession = false)]
    public ChangedMember[] GetChangedMembers(string securityKey, DateTime changedSince)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        List<ChangedMember> list = new List<ChangedMember>();

        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[1];
            param[0] = new SqlParameter("@ChangedSince", SqlDbType.VarChar);
            param[0].Value = changedSince;

            using (SqlDataReader reader = server.ExecuteReader(CommandType.StoredProcedure, "hlGetChangedMembers", param))
            {
                while (reader.Read())
                {
                    list.Add(new ChangedMember((string)reader["MemberKey"], (DateTime)reader["LastUpdated"]));
                }
                reader.Close();
            }
        }
        return list.ToArray();
    }

    [WebMethod(Description = "Gets all of the members of a given security group. Security groups are things like STAFF and IsMember.", EnableSession = false)]
    public GroupMember[] GetSecurityGroupMembers(string securityKey, string groupKey)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        List<GroupMember> list = new List<GroupMember>();

        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[1];
            param[0] = new SqlParameter("@GroupKey", SqlDbType.VarChar);
            param[0].Value = groupKey;

            using (SqlDataReader reader = server.ExecuteReader(CommandType.StoredProcedure, "hlGetSecurityGroupMembers", param))
            {
                while (reader.Read())
                {
                    string memberKey = string.Empty;
                    string role = string.Empty;
                    DateTime sinceDate = DateTime.MinValue;
                    DateTime startDate = DateTime.MinValue;
                    DateTime endDate = DateTime.MinValue;

                    if (reader["MemberKey"] != DBNull.Value)
                        memberKey = (string)reader["MemberKey"];
                    if (reader["Role"] != DBNull.Value)
                        role = (string)reader["role"];
                    if (reader["SinceDate"] != DBNull.Value)
                        sinceDate = (DateTime)reader["SinceDate"];
                    if (reader["StartDate"] != DBNull.Value)
                        startDate = (DateTime)reader["StartDate"];
                    if (reader["EndDate"] != DBNull.Value)
                        endDate = (DateTime)reader["EndDate"];

                    list.Add(new GroupMember(memberKey, role, sinceDate, startDate, endDate));
                }
                reader.Close();
            }
        }
        return list.ToArray();
    }

    [WebMethod(Description = "Gets all of the members of a given community group. Community groups are things like chapters, committees, councils, and member types.", EnableSession = false)]
    public GroupMember[] GetCommunityGroupMembers(string securityKey, string groupKey, string groupType)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        List<GroupMember> list = new List<GroupMember>();

        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[2];
            param[0] = new SqlParameter("@GroupKey", SqlDbType.VarChar);
            param[0].Value = groupKey;
            param[1] = new SqlParameter("@GroupType", SqlDbType.VarChar);
            param[1].Value = groupType;

            using (SqlDataReader reader = server.ExecuteReader(CommandType.StoredProcedure, "hlGetCommunityGroupMembers", param))
            {
                while (reader.Read())
                {
                    string memberKey = string.Empty;
                    string role = string.Empty;
                    DateTime sinceDate = DateTime.MinValue;
                    DateTime startDate = DateTime.MinValue;
                    DateTime endDate = DateTime.MinValue;

                    if (reader["MemberKey"] != DBNull.Value)
                        memberKey = (string)reader["MemberKey"];
                    if (reader["Role"] != DBNull.Value)
                        role = (string)reader["role"];
                    if (reader["SinceDate"] != DBNull.Value)
                        sinceDate = (DateTime)reader["SinceDate"];
                    if (reader["StartDate"] != DBNull.Value)
                        startDate = (DateTime)reader["StartDate"];
                    if (reader["EndDate"] != DBNull.Value)
                        endDate = (DateTime)reader["EndDate"];

                    list.Add(new GroupMember(memberKey, role, sinceDate, startDate, endDate));
                }
                reader.Close();
            }
        }
        return list.ToArray();
    }

    [WebMethod(Description = "Gets all of the community groups to which a member currently belongs. Community groups are things like chapters, committees, councils, and member types.", EnableSession = false)]
    public MemberGroup[] GetMemberCommunityGroups(string securityKey, string memberKey)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        return GetMemberGroups(memberKey, "hlGetMemberCommunityGroups");
    }

    [WebMethod(Description = "Gets all of the security groups to which a member currently belongs. Security Groups are any groups which can provide security access to information in the social media system.", EnableSession = false)]
    public MemberGroup[] GetMemberSecurityGroups(string securityKey, string memberKey)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        return GetMemberGroups(memberKey, "hlGetMemberSecurityGroups");
    }

    [WebMethod(Description = "Gets the member's contact details, like name, address, email, phone, etc., activities, and any additional demographic details that should be synced to their online profile.", EnableSession = false)]
    public MemberDetails GetMemberDetails(string securityKey, string memberKey)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        MemberDetails details = new MemberDetails();

        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[1];
            param[0] = new SqlParameter("@MemberKey", SqlDbType.VarChar);
            param[0].Value = memberKey;

            using (SqlDataReader reader = server.ExecuteReader(CommandType.StoredProcedure, "hlGetMemberDetails", param))
            {
                while (reader.Read())
                {
                    if (reader["MemberKey"] != DBNull.Value)
                        details.MemberKey = (string) reader["MemberKey"];
                    if (reader["Prefix"] != DBNull.Value)
                        details.Prefix = (string) reader["Prefix"];
                    if (reader["FirstName"] != DBNull.Value)
                        details.FirstName = (string) reader["FirstName"];
                    if (reader["LastName"] != DBNull.Value)
                        details.LastName = (string) reader["LastName"];
                    if (reader["Suffix"] != DBNull.Value)
                        details.Suffix = (string) reader["Suffix"];
                    if (reader["Designation"] != DBNull.Value)
                        details.Designation = (string) reader["Designation"];
                    if (reader["InformalName"] != DBNull.Value)
                        details.InformalName = (string) reader["InformalName"];
                    if (reader["CompanyName"] != DBNull.Value)
                        details.CompanyName = (string) reader["CompanyName"];
                    if (reader["Title"] != DBNull.Value)
                        details.Title = (string) reader["Title"];
                    if (reader["Email1"] != DBNull.Value)
                        details.Email1 = (string) reader["Email1"];
                    if (reader["Email2"] != DBNull.Value)
                        details.Email2 = (string) reader["Email2"];
                    if (reader["Phone1"] != DBNull.Value)
                        details.Phone1 = (string) reader["Phone1"];
                    if (reader["Phone1Type"] != DBNull.Value)
                        details.Phone1Type = (string) reader["Phone1Type"];
                    if (reader["Phone2"] != DBNull.Value)
                        details.Phone2 = (string) reader["Phone2"];
                    if (reader["Phone2Type"] != DBNull.Value)
                        details.Phone2Type = (string) reader["Phone2Type"];
                    if (reader["Phone3"] != DBNull.Value)
                        details.Phone3 = (string) reader["Phone3"];
                    if (reader["Phone3Type"] != DBNull.Value)
                        details.Phone3Type = (string) reader["Phone3Type"];
                    if (reader["Phone4"] != DBNull.Value)
                        details.Phone4 = (string) reader["Phone4"];
                    if (reader["Phone4Type"] != DBNull.Value)
                        details.Phone4Type = (string) reader["Phone4Type"];
                    if (reader["WebsiteUrl"] != DBNull.Value)
                        details.WebsiteUrl = (string) reader["WebsiteUrl"];
                    if (reader["PictureUrl"] != DBNull.Value)
                        details.PictureUrl = (string)reader["PictureUrl"];
                    if (reader["DoNotList"] != DBNull.Value)
                        details.DoNotList = (bool)reader["DoNotList"];
					if (reader["DoNotEmail"] != DBNull.Value)
                        details.DoNotEmail = (bool)reader["DoNotEmail"];
                    if (reader["Status"] != DBNull.Value)
                        details.Status = (string)reader["Status"];
                    if (reader["IsCompany"] != DBNull.Value)
                        details.IsCompany = (bool)reader["IsCompany"];
                    if (reader["CompanyID"] != DBNull.Value)
                        details.CompanyID = (string)reader["CompanyID"];
                    if (reader["Custom01"] != DBNull.Value)
                        details.Custom01 = (string) reader["Custom01"];
                    if (reader["Custom02"] != DBNull.Value)
                        details.Custom02 = (string) reader["Custom02"];
                    if (reader["Custom03"] != DBNull.Value)
                        details.Custom03 = (string) reader["Custom03"];
                    if (reader["Custom04"] != DBNull.Value)
                        details.Custom04 = (string) reader["Custom04"];
                    if (reader["Custom05"] != DBNull.Value)
                        details.Custom05 = (string) reader["Custom05"];
                    if (reader["Custom06"] != DBNull.Value)
                        details.Custom06 = (string) reader["Custom06"];
                    if (reader["Custom07"] != DBNull.Value)
                        details.Custom07 = (string) reader["Custom07"];
                    if (reader["Custom08"] != DBNull.Value)
                        details.Custom08 = (string) reader["Custom08"];
                    if (reader["Custom09"] != DBNull.Value)
                        details.Custom09 = (string) reader["Custom09"];
                    if (reader["Custom10"] != DBNull.Value)
                        details.Custom10 = (string) reader["Custom10"];
                }
                reader.Close();
            }

            SqlParameter[] param2 = new SqlParameter[1];
            param2[0] = new SqlParameter("@MemberKey", SqlDbType.VarChar);
            param2[0].Value = memberKey;

            List<MemberAddress> addresses = new List<MemberAddress>();

            using (SqlDataReader reader = server.ExecuteReader(CommandType.StoredProcedure, "hlGetMemberAddresses", param2))
            {
                while (reader.Read())
                {
                    MemberAddress address = new MemberAddress();
                    if (reader["AddressType"] != DBNull.Value)
                        address.AddressType = (string)reader["AddressType"];
                    if (reader["Address1"] != DBNull.Value)
                        address.Address1 = (string)reader["Address1"];
                    if (reader["Address2"] != DBNull.Value)
                        address.Address2 = (string)reader["Address2"];
                    if (reader["Address3"] != DBNull.Value)
                        address.Address3 = (string)reader["Address3"];
                    if (reader["City"] != DBNull.Value)
                        address.City = (string)reader["City"];
                    if (reader["StateProvince"] != DBNull.Value)
                        address.StateProvince = (string)reader["StateProvince"];
                    if (reader["PostalCode"] != DBNull.Value)
                        address.PostalCode = (string)reader["PostalCode"];
                    if (reader["CountryCode"] != DBNull.Value)
                        address.CountryCode = (string)reader["CountryCode"];

                    if (reader["Phone"] != DBNull.Value)
                    {
                        address.Phone1 = (string) reader["Phone"];
                        address.Phone1Type = address.AddressType;
                    }
                    if (reader["Fax"] != DBNull.Value)
                    {
                        address.Phone2 = (string) reader["Fax"];
                        address.Phone2Type = "Fax";
                    }
                    if (reader["TollFree"] != DBNull.Value)
                    {
                        address.Phone3 = (string) reader["TollFree"];
                        address.Phone3Type = "TollFree";
                    }

                    if (reader["EmailAddress"] != DBNull.Value)
                        address.EmailAddress = (string)reader["EmailAddress"];
                    if (reader["IsPreferredMail"] != DBNull.Value)
                        address.IsPreferredMail = (bool)reader["IsPreferredMail"];
                    if (reader["IsPreferredBill"] != DBNull.Value)
                        address.IsPreferredBill = (bool)reader["IsPreferredBill"];
                    if (reader["IsPreferredList"] != DBNull.Value)
                        address.IsPreferredList = (int)reader["IsPreferredList"] == 1;
                    if (reader["IsListCityState"] != DBNull.Value)
                        address.IsListCityState = (int)reader["IsListCityState"] == 1;
                    if (reader["LegacyAddressKey"] != DBNull.Value)
                        address.LegacyAddressKey = (string)reader["LegacyAddressKey"];

                    addresses.Add(address);
                }
                reader.Close();
            }
            details.Addresses = addresses.ToArray();

            // for compatibility with older versions of the provider
            if (details.Addresses.Length > 0)
            {
                details.Address1 = details.Addresses[0].Address1;
                details.Address2 = details.Addresses[0].Address2;
                details.Address3 = details.Addresses[0].Address3;
                details.City = details.Addresses[0].City;
                details.StateProvince = details.Addresses[0].StateProvince;
                details.PostalCode = details.Addresses[0].PostalCode;
                details.CountryCode = details.Addresses[0].CountryCode;
            }
        }
        return details;

        /*        if (!iboAdmin.IsSystemInitialized)
                    iboAdmin.InitializeSystem();
                MemberDetails details = new MemberDetails();
                CContact contact = new CContact(iMISUser, memberKey);
                details.MemberKey = memberKey;
                details.Prefix = contact.Prefix;
                details.FirstName = contact.FirstName;
                details.LastName = contact.LastName;
                details.Suffix = contact.Suffix;
                details.Designation = contact.Designation;
                details.InformalName = contact.Informal;
                details.CompanyName = contact.InstituteName;
                details.Title = contact.Title;
                details.Email1 = contact.EmailAddress;
                details.Email2 = string.Empty;
                details.Phone1 = contact.WorkPhone;
                details.Phone1Type = "Work";
                details.Phone2 = contact.HomePhone;
                details.Phone2Type = "Home";
                details.Phone3 = contact.Fax;
                details.Phone3Type = "FAX";
                details.Phone4 = contact.TollFreePhone;
                details.Phone4Type = "TollFree";
                details.Address1 = contact.DefaultAddress.Address1;
                details.Address2 = contact.DefaultAddress.Address2;
                details.Address3 = contact.DefaultAddress.Address3;
                details.City = contact.DefaultAddress.City;
                details.StateProvince = contact.DefaultAddress.StateProvince;
                details.PostalCode = contact.DefaultAddress.PostalCode;
                details.CountryCode = contact.DefaultAddress.Country;
                details.WebsiteUrl = contact.WebsiteAddress;
        */
        /*
        List<DemographicField> demographics = new List<DemographicField>();
        foreach (KeyValuePair<string, List<string>> entity in DemographicFields)
        {
            foreach (CExtTable table in contact.ExtTables)
            {
                if (table.TableName.Equals(entity.Key,StringComparison.InvariantCultureIgnoreCase))
                {
                    for (int i = 0, ic = table.InstancesCount; i < ic; i++)
                    {
                        foreach (string property in entity.Value)
                        {
                            CExtTable instance = table.GetInstance(i);
                            CExtField field = instance.GetField(property);
                            if (field != null)
                                demographics.Add(
                                    new DemographicField(i.ToString(), entity.Key, property, field.FieldValue.ToString()));
                        }
                    }
                }
            } 
        }
        details.Demographics = demographics.ToArray();

        if (contact.Activities != null)
        {
            List<Activity> activities = new List<Activity>();
            foreach (string activityType in ReadActivityTypes)
            {
                foreach (CActivity activity in contact.Activities)
                    if (activity.ActivityType.Equals(activityType, StringComparison.InvariantCultureIgnoreCase))
                        activities.Add(
                            new Activity(activity.SequenceNumber.ToString(), activity.ActivityType, activity.ProductCode,
                                         activity.TransactionDate, activity.Description, string.Empty,
                                         activity.Quantity.ToString(), activity.Amount.ToString()));
            }
            details.Activities = activities.ToArray();
        }

        if (contact.Subscriptions != null)
        {
            List<Subscription> subscriptions = new List<Subscription>();
            foreach (string subscriptionType in ReadSubscriptionTypes)
            {
                foreach (CSubscription subscription in contact.Subscriptions)
                    if (
                        subscription.ProductTypeCode.Equals(subscriptionType,
                                                            StringComparison.InvariantCultureIgnoreCase))
                        subscriptions.Add(
                            new Subscription(subscription.ProductCode, subscription.ProductTypeCode,
                                             subscription.DuesProduct.Title, subscription.ContinuousSince,
                                             subscription.PaidBeginDate, subscription.PaidThruDate));
            }
            details.Subscriptions = subscriptions.ToArray();
        }
        */
        //        return details;
    }

    [WebMethod(Description = "Gets the meta data for the specified user defined table.", EnableSession = false)]
    public UDItemMeta GetUserDefinedMeta(string securityKey, string tableName)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        SqlParameter[] param;
        string sql;
        UDItemMeta meta = null;

        sql = @"select t.TABLE_NAME TableName, t.DESCRIPTION TableDescription, t.ALLOW_MULTIPLE_INSTANCES AllowMultiple,
                           f.FIELD_NAME FieldName, f.FIELD_TYPE FieldType, f.FIELD_LENGTH [Length], cast(DECIMAL_POSITIONS as int) [Scale], f.PROMPT Prompt, f.VALIDATION_TABLE LookupName, f.MULTI_SELECT MultiSelect
                     from  UD_Table t
                           inner join UD_Field f on f.TABLE_NAME = t.TABLE_NAME
                     where t.TABLE_NAME = @TableName
                     order by f.SEQ";
        param = new SqlParameter[1];
        param[0] = new SqlParameter("@TableName", SqlDbType.VarChar);
        param[0].Value = tableName;

        using (DataServer server = new DataServer())
        {
            using (SqlDataReader reader = server.ExecuteReader(CommandType.Text, sql, param))
            {
                while (reader.Read())
                {
                    if (meta == null)
                        meta = new UDItemMeta((string)reader["TableName"], (string)reader["TableDescription"], (bool)reader["AllowMultiple"]);
                    meta.AddValueMeta((string)reader["FieldName"], (string)reader["FieldType"], (int)reader["Length"], (int)reader["Scale"], (string)reader["Prompt"], (string)reader["LookupName"], (bool)reader["MultiSelect"]);
                }
                reader.Close();
            }
        }

        return meta;
    }

    [WebMethod(Description = "Gets the data for the specified ID from the specified user defined table.", EnableSession = false)]
    public UDData GetUserDefinedData(string securityKey, string id, string tableName, string fieldList)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        bool allowMultiple = false;
        DataTable schema = null;
        SqlParameter[] param;
        string columnName;
        string sql;
        UDData data = new UDData(tableName);
        UDItem item;

        using (DataServer server = new DataServer())
        {
            sql = "select ALLOW_MULTIPLE_INSTANCES from UD_Table where TABLE_NAME = @TableName";
            param = new SqlParameter[1];
            param[0] = new SqlParameter("@TableName", SqlDbType.VarChar);
            param[0].Value = tableName;
            using (SqlDataReader reader = server.ExecuteReader(CommandType.Text, sql, param))
            {
                if (reader.Read())
                    allowMultiple = (bool)reader[0];
                else
                    throw new ApplicationException("Invalid Table Name. " + tableName + " Does Not Exist");
                reader.Close();
            }

            param = new SqlParameter[1];
            param[0] = new SqlParameter("@ID", SqlDbType.VarChar);
            param[0].Value = id;
            if (allowMultiple)
            {
                sql = "select " + (string.IsNullOrEmpty(fieldList) ? "*" : "ID,SEQN," + fieldList) +
                      " from  " + tableName +
                      " where ID = @ID" +
                      " order by SEQN";
            }
            else
            {
                sql = "select " + (string.IsNullOrEmpty(fieldList) ? "*" : "ID," + fieldList) +
                      " from  " + tableName +
                      " where ID = @ID";
            }
            using (SqlDataReader reader = server.ExecuteReader(CommandType.Text, sql, param))
            {
                while (reader.Read())
                {
                    if (allowMultiple)
                        item = data.AddItem((string)reader["ID"], (int)reader["SEQN"]);
                    else
                        item = data.AddItem((string)reader["ID"]);
                    if (schema == null)
                        schema = reader.GetSchemaTable();
                    foreach (DataRow row in schema.Rows)
                    {
                        columnName = (string)row["ColumnName"];
                        if (columnName != "ID" && columnName != "SEQN")
                            item.AddColumn(columnName, reader[columnName]);
                    }
                }
                reader.Close();
            }
        }

        return data;
    }

    [WebMethod(Description = "Writes the member's contact details, like name, address, email, phone, etc., activities, and any additional demographic details that was changed on their online profile back to the database.", EnableSession = false)]
    public void SetMemberDetails(string securityKey, MemberDetails details)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");
        if (!iboAdmin.IsSystemInitialized)
            iboAdmin.InitializeSystem();

        CContact contact = new CContact(iMISUser, details.MemberKey);

        if (!details.Prefix.Equals(contact.Prefix))
            contact.Prefix = details.Prefix;
        if (!details.FirstName.Equals(contact.FirstName))
            contact.FirstName = details.FirstName;
        if (!details.LastName.Equals(contact.LastName))
            contact.LastName = details.LastName;
        if (!details.Suffix.Equals(contact.Suffix))
            contact.Suffix = details.Suffix;
        if (!details.Designation.Equals(contact.Designation))
            contact.Designation = details.Designation;
        if (!details.InformalName.Equals(contact.Informal))
            contact.Informal = details.InformalName;
        if (!details.CompanyName.Equals(contact.InstituteName))
            contact.InstituteName = details.CompanyName;
        if (!details.Title.Equals(contact.Title))
            contact.Title = details.Title;
        if (!details.Email1.Equals(contact.EmailAddress))
            contact.EmailAddress = details.Email1;
        if (!details.Phone1.Equals(contact.WorkPhone))
            contact.WorkPhone = details.Phone1;
        if (!details.Phone2.Equals(contact.HomePhone))
            contact.HomePhone = details.Phone2;
        if (!details.Phone3.Equals(contact.Fax))
            contact.Fax = details.Phone3;
        if (!details.Phone4.Equals(contact.TollFreePhone))
            contact.TollFreePhone = details.Phone4;
        if (!details.WebsiteUrl.Equals(contact.WebsiteAddress))
            contact.WebsiteAddress = details.WebsiteUrl;

        foreach (MemberAddress memberAddress in details.Addresses)
        {
            CAddress address = null;
            if (!string.IsNullOrEmpty(memberAddress.LegacyAddressKey))
            {
                int addressNum;
                if (int.TryParse(memberAddress.LegacyAddressKey, out addressNum))
                    address = contact.GetAddressByNum(addressNum);
            }
            else
                address = contact.NewAddress();

            if (address != null)
            {
                address.AddressPurpose = memberAddress.AddressType;
                address.Address1 = memberAddress.Address1;
                address.Address2 = memberAddress.Address2;
                address.Address3 = memberAddress.Address3;
                address.City = memberAddress.City;
                address.StateProvince = memberAddress.StateProvince;
                address.PostalCode = memberAddress.PostalCode;
                address.Country = memberAddress.CountryCode;
                address.Phone = memberAddress.Phone1;
                address.Fax = memberAddress.Phone2;
                address.TollFree = memberAddress.Phone3;
                address.EmailAddress = memberAddress.EmailAddress;
                if (!address.IsPreferredMail && memberAddress.IsPreferredMail)
                    address.IsPreferredMail = memberAddress.IsPreferredMail;
                if (!address.IsPreferredBill && memberAddress.IsPreferredBill)
                    address.IsPreferredBill = memberAddress.IsPreferredBill;
            }
        }

        contact.Save();

        // save custom data, if applicable
        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[6];
            param[0] = new SqlParameter("@ID", SqlDbType.VarChar);
            param[0].Value = details.MemberKey;
            param[1] = new SqlParameter("@Custom01", SqlDbType.VarChar);
            param[1].Value = details.Custom01;
            param[2] = new SqlParameter("@Custom02", SqlDbType.VarChar);
            param[2].Value = details.Custom02;
            param[3] = new SqlParameter("@Custom03", SqlDbType.VarChar);
            param[3].Value = details.Custom03;
            param[4] = new SqlParameter("@Custom04", SqlDbType.VarChar);
            param[4].Value = details.Custom04;
            param[5] = new SqlParameter("@Custom05", SqlDbType.VarChar);
            param[5].Value = details.Custom05;

            server.ExecuteNonQuery(CommandType.StoredProcedure, "hlSetCustomData", param);
        }
    }

    [WebMethod(Description = "Writes a Activity to the activity table for a member and retruns the activity seqn", EnableSession = false)]
    public int WriteActivityDetails(string securityKey, string memberKey, string activityType, DateTime activityDate, string Description)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        // save activity data and get back the activity seqn
        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[5];
            param[0] = new SqlParameter("@MemberKey", SqlDbType.VarChar);
            param[0].Value = memberKey;
            param[1] = new SqlParameter("@ActivityType", SqlDbType.VarChar);
            param[1].Value = activityType;
            param[2] = new SqlParameter("@ActivityDate", SqlDbType.VarChar);
            param[2].Value = activityDate;
            param[3] = new SqlParameter("@Description", SqlDbType.VarChar);
            param[3].Value = Description;
            param[4] = new SqlParameter("@seqn", SqlDbType.Int)
            {
                Direction = ParameterDirection.Output
            };


            server.ExecuteNonQuery(CommandType.StoredProcedure, "hlWriteActivity", param);
            int Activityseqn = Convert.ToInt32(param[4].Value);
            return Activityseqn;
        }
    }

    [WebMethod(Description = "Gets the details of all future Events so that they can be brought into the calendar. ", EnableSession = false)]
    public Meeting[] GetEvents(string securityKey)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");

        List<Meeting> meetings = new List<Meeting>();

        using (DataServer server = new DataServer())
        {
            using (SqlDataReader reader = server.ExecuteReader(CommandType.Text, "SELECT * FROM Meet_Master WHERE END_DATE > GetDate()"))
            {
                while (reader.Read())
                {
                    Meeting meeting = RetrieveEvent(reader);
                    meetings.Add(meeting);
                }
                reader.Close();
            }
        }
        return meetings.ToArray();
    }

    public Meeting GetEvent(string securityKey, string legacyEventKey)
    {
        if (!CheckSecurity(securityKey))
            throw new AccessViolationException("Unauthorized Access.");
        
        Meeting? meeting = null;
        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[1];
            param[0] = new SqlParameter("@Meeting", SqlDbType.VarChar);
            param[0].Value = legacyEventKey;
            using (SqlDataReader reader = server.ExecuteReader(CommandType.Text, "SELECT * FROM Meet_Master WHERE MEETING = @Meeting"))
            {
                while (reader.Read())
                {
                    meeting = RetrieveEvent(reader);
                    break;
                }
                reader.Close();
            }
        }
        if (meeting != null)
            return (Meeting) meeting;
        throw new ArgumentException("Meeting with key '" + legacyEventKey + "' not found.");
    }

    public Meeting RetrieveEvent(SqlDataReader reader)
    {
        Meeting meeting = new Meeting();
        if (reader["MEETING"] != DBNull.Value)
            meeting.LegacyEventKey = (string)reader["MEETING"];
        if (reader["TITLE"] != DBNull.Value)
            meeting.EventTitle = (string)reader["TITLE"];
        if (reader["MEETING_TYPE"] != DBNull.Value)
            meeting.EventTypeKey = (string)reader["MEETING_TYPE"];
        if (reader["DESCRIPTION"] != DBNull.Value)
            meeting.EventDescription = (string)reader["DESCRIPTION"];
        if (reader["BEGIN_DATE"] != DBNull.Value)
            meeting.StartDateTime = (DateTime)reader["BEGIN_DATE"];
        if (reader["END_DATE"] != DBNull.Value)
            meeting.EndDateTime = (DateTime)reader["END_DATE"];
        if (reader["ADDRESS_1"] != DBNull.Value)
            meeting.Address1 = (string)reader["ADDRESS_1"];
        if (reader["ADDRESS_2"] != DBNull.Value)
            meeting.Address2 = (string)reader["ADDRESS_2"];
        if (reader["ADDRESS_3"] != DBNull.Value)
            meeting.Address3 = (string)reader["ADDRESS_3"];
        if (reader["CITY"] != DBNull.Value)
            meeting.City = (string)reader["CITY"];
        if (reader["STATE_PROVINCE"] != DBNull.Value)
            meeting.StateProvinceCode = (string)reader["STATE_PROVINCE"];
        if (reader["ZIP"] != DBNull.Value)
            meeting.PostalCode = (string)reader["ZIP"];
        if (reader["COUNTRY"] != DBNull.Value)
            meeting.CountryCode = (string)reader["COUNTRY"];
        if (reader["EARLY_CUTOFF"] != DBNull.Value)
            meeting.EarlyRegistrationDate = (DateTime)reader["EARLY_CUTOFF"];
        if (reader["REG_CUTOFF"] != DBNull.Value)
            meeting.RegularRegistrationDate = (DateTime)reader["REG_CUTOFF"];
        if (reader["LATE_CUTOFF"] != DBNull.Value)
            meeting.LastRegistrationDate = (DateTime)reader["LATE_CUTOFF"];
        if (reader["MEETING_URL"] != DBNull.Value)
            meeting.DetailURL = (string)reader["MEETING_URL"];
        if (reader["STATUS"] != DBNull.Value)
            meeting.Status = ((bool)reader["WEB_ENABLED"] && ((string)reader["STATUS"]) == "A") ? "A" : "D";
        meeting.Functions = GetEventFunctions(meeting.LegacyEventKey);
        return meeting;
    }

    private MeetingFunction[] GetEventFunctions(string legacyEventKey)
    {
        List<MeetingFunction> functions = new List<MeetingFunction>();

        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[1];
            param[0] = new SqlParameter("@Meeting", SqlDbType.VarChar);
            param[0].Value = legacyEventKey;
            using (SqlDataReader reader = server.ExecuteReader(CommandType.Text, "select a.PRODUCT_CODE, a.TITLE, b.FUNCTION_TYPE, a.DESCRIPTION, a.WEB_DESC, b.BEGIN_DATE_TIME, b.END_DATE_TIME, b.CEU_AMOUNT, b.COURSE_CODE, a.STATUS, b.WEB_ENABLED FROM Product a inner join Product_Function b on a.PRODUCT_CODE = b.PRODUCT_CODE WHERE a.PRODUCT_MAJOR = @Meeting", param))
            {
                while (reader.Read())
                {
                    MeetingFunction function = new MeetingFunction();
                    if (reader["PRODUCT_CODE"] != DBNull.Value)
                        function.LegacyOptionKey = (string)reader["PRODUCT_CODE"];
                    if (reader["TITLE"] != DBNull.Value)
                        function.FunctionTitle = (string)reader["TITLE"];
                    if (reader["FUNCTION_TYPE"] != DBNull.Value)
                        function.FunctionTypeKey = (string)reader["FUNCTION_TYPE"];
                    if (reader["WEB_DESC"] != DBNull.Value)
                        function.FunctionDescription = (string)reader["WEB_DESC"];
                    else if (reader["DESCRIPTION"] != DBNull.Value)
                        function.FunctionDescription = (string)reader["DESCRIPTION"];
                    if (reader["BEGIN_DATE_TIME"] != DBNull.Value)
                        function.StartDateTime = (DateTime)reader["BEGIN_DATE_TIME"];
                    if (reader["END_DATE_TIME"] != DBNull.Value)
                        function.EndDateTime = (DateTime)reader["END_DATE_TIME"];
                    if (reader["CEU_AMOUNT"] != DBNull.Value)
                        function.CECredits = ((decimal)reader["CEU_AMOUNT"]).ToString();
                    if (reader["COURSE_CODE"] != DBNull.Value)
                        function.SessionCode = (string)reader["COURSE_CODE"];
                    if (reader["STATUS"] != DBNull.Value)
                        function.Status = ((bool)reader["WEB_ENABLED"] && ((string)reader["STATUS"]) == "A") ? "A" : "D";
                    functions.Add(function);
                }
                reader.Close();
            }
        }
        return functions.ToArray();
    }

    // executes the given stored procedure and returns the results in an array of MemberGroup objects
    private static MemberGroup[] GetMemberGroups(string memberKey, string procedureName)
    {
        List<MemberGroup> list = new List<MemberGroup>();

        using (DataServer server = new DataServer())
        {
            SqlParameter[] param = new SqlParameter[1];
            param[0] = new SqlParameter("@MemberKey", SqlDbType.VarChar);
            param[0].Value = memberKey;

            using (SqlDataReader reader = server.ExecuteReader(CommandType.StoredProcedure, procedureName, param))
            {
                while (reader.Read())
                {
                    string groupKey = string.Empty;
                    string groupName = string.Empty;
                    string groupType = string.Empty;
                    string role = string.Empty;
                    DateTime sinceDate = DateTime.MinValue;
                    DateTime startDate = DateTime.MinValue;
                    DateTime endDate = DateTime.MinValue;

                    if (reader["GroupKey"] != DBNull.Value)
                        groupKey = (string)reader["GroupKey"];
                    if (reader["GroupName"] != DBNull.Value)
                        groupName = (string)reader["GroupName"];
                    if (reader["GroupType"] != DBNull.Value)
                        groupType = (string)reader["GroupType"];
                    if (reader["Role"] != DBNull.Value)
                        role = (string)reader["role"];
                    if (reader["SinceDate"] != DBNull.Value)
                        sinceDate = (DateTime)reader["SinceDate"];
                    if (reader["StartDate"] != DBNull.Value)
                        startDate = (DateTime)reader["StartDate"];
                    if (reader["EndDate"] != DBNull.Value)
                        endDate = (DateTime)reader["EndDate"];

                    list.Add(new MemberGroup(groupKey, groupName, groupType, role, sinceDate, startDate, endDate));
                }
                reader.Close();
            }
        }
        return list.ToArray();
    }

    private static IiMISUser iMISUser
    {
        get
        {
            if (_iMISUser == null)
            {
                _iMISUser = CStaffUser.Login(ConfigurationManager.AppSettings["System.WebLogin"], ConfigurationManager.AppSettings["System.WebPassword"]);
                _iMISUser.ThrowExceptionOnError = true;
            }
            return _iMISUser;
        }
    }

    private static Dictionary<string, List<string>> DemographicFields
    {
        get
        {
            if (_DemographicFields == null)
            {
                _DemographicFields = new Dictionary<string, List<string>>();
                string demoFields = ConfigurationManager.AppSettings["ReadDemographicsFields"];
                if (demoFields != null && demoFields.Length > 0)
                {
                    string[] fields = demoFields.Split(',');
                    foreach (string field in fields)
                    {
                        string[] parts = field.Split('.');
                        if (parts.Length == 2)
                        {
                            if (_DemographicFields.ContainsKey(parts[0]))
                                _DemographicFields[parts[0]].Add(parts[1]);
                            else
                            {
                                List<string> columns = new List<string>();
                                columns.Add(parts[1]);
                                _DemographicFields.Add(parts[0], columns);
                            }
                        }
                    }
                }
            }
            return _DemographicFields;
        }
    }

    private static string[] ReadActivityTypes
    {
        get
        {
            if (_ReadActivityTypes == null)
                _ReadActivityTypes = ConfigurationManager.AppSettings["ReadActivityTypes"].Split(',');
            return _ReadActivityTypes;
        }
    }

    private static string[] ReadSubscriptionTypes
    {
        get
        {
            if (_ReadSubscriptionTypes == null)
                _ReadSubscriptionTypes = ConfigurationManager.AppSettings["ReadSubscriptionTypes"].Split(',');
            return _ReadSubscriptionTypes;
        }
    }
}

