﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using System.Security.Cryptography;
using System.Text;

namespace RealMagnet.TrackingAPI
{
	/// <summary>
	/// Severity of a status message.
	/// </summary>
	public enum Severity
	{
		Error = 1,
		Warning = 2,
		Information = 3,
	}

	/// <summary>
	/// Specifies the new tracking state for a particular type of tracking.
	/// </summary>
	/// <remarks>If both StartId & StartDate are specified, StartId takes precedence.</remarks>
	[DataContract(Namespace="http://xmlns.realmagnet.com/magnetmail-api-service/v5/data")]
	public class TrackingStateRequest
	{
		/// <summary>
		/// The type of tracking for which the state should be updated.
		/// </summary>
		[DataMember(IsRequired=false, EmitDefaultValue=false)]
		public string TrackingType { get; set; }

		/// <summary>
		/// The ID of the tracking record to be considered the next unread record
		/// </summary>
		[DataMember(Name="starting-id", IsRequired=false, EmitDefaultValue=false)]
		public long? StartId { get; set; }

		/// <summary>
		/// The date from which point all records will be considered unread
		/// </summary>
		[DataMember(Name="starting-date", IsRequired=false, EmitDefaultValue=false)]
		public DateTime? StartDate { get; set; }
	}

    /// <summary>
	/// Specifies the new tracking state for a particular type of tracking.
	/// </summary>
	/// <remarks>If both StartId & StartDate are specified, StartId takes precedence.</remarks>
	[DataContract(Namespace = "http://xmlns.realmagnet.com/magnetmail-api-service/v5/data")]
    public class TrackingStateRestRequest
    {
        /// <summary>
        /// The type of tracking for which the state should be updated.
        /// </summary>
        [DataMember(IsRequired = false, EmitDefaultValue = false)]
        public string TrackingType { get; set; }

        /// <summary>
        /// The ID of the tracking record to be considered the next unread record
        /// </summary>
        [DataMember(Name = "starting-id", IsRequired = false, EmitDefaultValue = false)]
        public long? StartId { get; set; }

        /// <summary>
        /// The date from which point all records will be considered unread
        /// </summary>
        [DataMember(Name = "starting-date", IsRequired = false, EmitDefaultValue = false)]
        public string StartDate { get; set; }
    }

    /// <summary>
    /// Results of a tracking date request or state update operation.
    /// </summary>
    [DataContract(Namespace="http://xmlns.realmagnet.com/magnetmail-api-service/v5/data")]
	public class TrackingResponse
	{
		/// <summary>
		/// The value of starting-id to be used in a subsequent call to GetTrackingDataInRange
		/// </summary>
		[DataMember(Name="next-start-id")]
		public long NextStartId { get; set; }

		/// <summary>
		/// Ordered list of field names included in the results
		/// </summary>
        [DataMember(Name = "fields", IsRequired = false)]
		public string[] Fields { get; set; }

		/// <summary>
		/// Each record in the array corresponds to one tracking record, and each element in one records is the value for the corresponding field (named in Fields)
		/// </summary>
        [DataMember(Name = "data", IsRequired = false)]
		public string[][] Data { get; set; }

		/// <summary>
		/// If true, make another request to get more data.
		/// </summary>
		[DataMember(Name="has-more", IsRequired=false)]
		public bool HasMore { get; set; }

		/// <summary>
		/// Diagnostic information such as error messages and warnings.
		/// </summary>
		[DataMember(IsRequired = true)]
		public StatusMessage[] Messages { get; set; }
	}

	/// <summary>
	/// Diagnostic information such as error messages and warnings.
	/// </summary>
	[DataContract]
	public class StatusMessage
	{
		/// <summary>
		/// 1 => Error; 2 => Warning; 3 => Information.
		/// </summary>
		[DataMember(IsRequired = true)]
		public int Severity;

		/// <summary>
		/// Message text
		/// </summary>
		[DataMember(IsRequired = true)]
		public string Message;
	}

	/// <summary>
	///  Real Magnet UploadAndSendAPI Client. Use this class to access the API.
	/// </summary>
	public class TrackingApiClient
	{
		public string Url = @"http://api105.magnetmail.net/v5/rest/tracking/";
		public string SecretKey = null;
		public string MailUserId = null;
		public int? Timeout = null;

		/// <summary>
		/// Initializes a new instance of the TrackingApiClient class, using the specified credentials.
		/// </summary>
		/// <param name="mailUser">The acronym of your account as provided by Real Magnet.</param>
		/// <param name="key">The API secret key as provided by Real Magnet.</param>
		public TrackingApiClient( string mailUser, string key )
		{
			this.MailUserId = mailUser;
			this.SecretKey = key;
		}

		/// <summary>
		/// Get the next unread tracking records and advance the tracking state.
		/// </summary>
		/// <param name="tracking_type">The type of tracking records to retreive</param>
		/// <param name="max_rows">Optional; The maximum number of records to retrieve</param>
		/// <param name="fields">Optional; limits the fields named in the response to these.</param>
		/// <returns></returns>
		public TrackingResponse StreamTrackingData(string tracking_type, int? max_rows = null, params string[] fields)
		{
			return SendRequest<object, TrackingResponse>(FormatUrl(tracking_type + "/next/", null, null, null, max_rows, fields));
		}

		/// <summary>
		/// Get tracking records with a specified starting point, and leave the tracking state untouched.
		/// </summary>
		/// <param name="tracking_type">The type of tracking records to retreive</param>
		/// <param name="starting_id">Start at the record with this ID</param>
		/// <param name="starting_on">Start with the first record of the specified type on this date (time ignored)</param>
		/// <param name="ending_on">Optional; tracking on this date will be included in the results, but tracking after this date will not be included.</param>
		/// <param name="max_rows">Optional; The maximum number of records to retrieve</param>
		/// <param name="fields">Optional; limits the fields named in the response to these.</param>
		/// <remarks>Only one of starting_id or starting_on is required; if both are specified, starting_id takes precedences</remarks>
		public TrackingResponse GetTrackingDataInRange(string tracking_type, long? starting_id = null, DateTime? starting_on = null, DateTime? ending_on = null, int? max_rows = null, params string[] fields)
		{
			if (!starting_id.HasValue && !starting_on.HasValue)
				throw new ArgumentException("starting_id or starting_on required");

			return SendRequest<object, TrackingResponse>(FormatUrl(tracking_type + "/", starting_id, starting_on, ending_on, max_rows, fields));
		}

		/// <summary>
		/// Update the tracking state (used by StreamTrackingData) to the specified ID or date.
		/// </summary>
		/// <param name="tracking_type">The type of tracking records for which the state will be updated</param>
		/// <param name="starting_id">On the next StreamTrackingData call, start at the record with this ID</param>
		/// <param name="starting_on">On the next StreamTrackingData call, start with the first record of the specified type on this date (time ignored)</param>
		/// <remarks>Only one of starting_id or starting_on is required; if both are specified, starting_id takes precedences.</remarks>
		public TrackingResponse UpdateTrackingStreamState(string tracking_type, long? starting_id = null, string starting_on = null)
		{
			if (!starting_id.HasValue && String.IsNullOrEmpty(starting_on))
				throw new ArgumentException("starting_id or starting_on required");

			return SendRequest<TrackingStateRestRequest, TrackingResponse>(tracking_type + "/", "PUT", new TrackingStateRestRequest
            {
					StartId = starting_id,
					StartDate = starting_on,
				});
		}

        /// <summary>
        ///  Sends a request of to the API passing a message of type RQ and retuning a message of type RS.
        ///  Note, that both RQ and RS MUST contain DataContract attributes to serialize properly to JSON
        ///     url - the relative RESTful URL of the method call
        ///     httpMethod - allows to specify the HTTP method (defaults to GET)
        /// </summary>    
        protected RS SendRequest<RQ, RS>( string url, string httpMethod = "GET", RQ request = null ) 
            where RS: class 
            where RQ: class
        {
            // create request
			var wrq = WebRequest.CreateHttp(Url + url);
			wrq.Timeout = this.Timeout ?? wrq.Timeout;

            // setup headers
            wrq.Method = httpMethod;
            wrq.Date = DateTime.Now;
			wrq.ContentType = "application/json";
            wrq.AllowAutoRedirect = true;

            // sign the request for auth purposes
            var toSign = CreateStringToSign( wrq );
            var signature = CreateSignature( toSign );
            wrq.Headers[HttpRequestHeader.Authorization] = string.Format( "RealMagnet {0}:{1}", MailUserId, signature );
            
            // serialize request data
            if( request != null )
            {
                var json = new DataContractJsonSerializer( typeof(RQ) );
                using( var st = wrq.GetRequestStream() )
                {
                    json.WriteObject( st, request );
                }
            }

            // de-serialize response
            using( var wrs = wrq.GetResponse() )
            {
                if( wrs.ContentLength > 0 )
                {
                    using( var st = wrs.GetResponseStream() )
                    {
                        var json = new DataContractJsonSerializer( typeof( RS ) );
                        return json.ReadObject( st ) as RS;
                    }
                }
            }
            
            return null;
        }

		/// <summary>
		///  Creates a fingerprint string for your request. The server requires that the string is normalized this way.
		/// </summary>    
		protected string CreateStringToSign( HttpWebRequest request )
		{
			const string CustomHeaderPrefix = "X-RealMagnet-";

			var headers = string.Join( "\n", (
					from key in request.Headers.AllKeys
					where key.StartsWith( CustomHeaderPrefix )
					orderby key
					select key.ToLowerInvariant() + ":" + string.Join( ",", request.Headers.GetValues( key ) )
					).ToArray()
				);

			return string.Format(
				"{0}\n{1}\n{2}\n{3}\n{4}\n{5}\n{6}",
				request.Method.ToUpperInvariant(),
				request.Headers[HttpRequestHeader.ContentMd5],
				request.ContentType,
				request.Headers[HttpRequestHeader.Date],
				headers,
				request.RequestUri.AbsoluteUri,
				"" // Action is always "" when using JSON.
				);
		}

		/// <summary>
		///  Creates a digital signature for your message fingerprint using HMAC SHA1
		/// </summary>    
		protected string CreateSignature( string stringToSign )
		{
			var encoding = new UTF8Encoding( false );
			var keyBytes = encoding.GetBytes( SecretKey );
			var messageBytes = encoding.GetBytes( stringToSign );
			var hmac = new HMACSHA1( keyBytes );
			var hash = hmac.ComputeHash( messageBytes );
			return Convert.ToBase64String( hash );
		}

		/// <summary>
		/// Create a URL for tracking
		/// </summary>
		private string FormatUrl( string url, long? starting_id, DateTime? starting_on, DateTime? ending_on, int? max_rows, string[] fields )
		{
			var sb = new StringBuilder();
			if (starting_id.HasValue)
				sb.Append("&starting-id=").Append(starting_id.Value);
			if (starting_on.HasValue)
				sb.Append("&starting-on=").Append(starting_on.Value.Date.ToString("yyyy-MM-dd", System.Globalization.CultureInfo.InvariantCulture));
			if (ending_on.HasValue)
				sb.Append("&ending-on=").Append(ending_on.Value.Date.ToString("yyyy-MM-dd", System.Globalization.CultureInfo.InvariantCulture));
			if (max_rows.HasValue)
				sb.Append("&max-rows=").Append(max_rows.Value);
			if (null != fields && 0 != fields.Length)
				sb.Append("&fields=").Append(System.Net.WebUtility.UrlEncode(string.Join(",", fields)));

			if (0 != sb.Length)
				sb.Remove(0, 1);

			return url + "?" + sb.ToString();
		}
	}
}
